package de.duehl.basics.logging;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;

/**
 * Dieses Klasse dient zum Loggen von Prozessabläufen im Hauptspeicher. Es wird vermerkt, wann und
 * von wo aufgerufen wird. Die Folgenden Informationen werden geloggt:
 *
 *     1: Datum
 *     2: Uhrzeit
 *     3: Klasse des Aufrufers
 *     4: Methode des Aufrufers
 *     5: Zeilennummer
 *     6: Inhalt der Nachricht
 *
 * @version 1.01     2017-03-31
 * @author Christian Dühl
 */

public class MemoryLogger extends StructuredLogger {

    /** Liste der geloggten Informationen im Speicher. */
    private final List<LogEntry> logEntries;

    /** Konstruktor. */
    public MemoryLogger() {
        super();
        logEntries = new ArrayList<>();
    }

    /**
     * Hier wird ein komplexerer Eintrag erzeugt mit sechs tabgetrennten Spalten.
     *     1: Datum
     *     2: Zeit
     *     3: Klasse des Aufrufers
     *     4: Methode des Aufrufers
     *     5: Zeilennummer
     *     6: Inhalt der Nachricht
     *
     * @param text
     *            Inhalt der Nachricht.
     */
    @Override
    public void log(String text) {
        log(text, 1);
        /*
         * Diese Methode kann man nicht einfach weglassen, sie ist zwar in StructuredLogger
         * vorhanden, aber dann stimmt der Offset nicht. Warum eigentlich nicht?
         */
    }

    /**
     * Hier wird ein komplexerer Eintrag erzeugt mit sechs tabgetrennten Spalten.
     *     1: Datum
     *     2: Zeit
     *     3: Klasse des Aufrufers
     *     4: Methode des Aufrufers
     *     5: Zeilennummer
     *     6: Inhalt der Nachricht
     *
     * @param text
     *            Inhalt der Nachricht.
     * @param stacktraceOffset
     *            Offset im Stacktrace zum Ermitteln des richtigen Aufrufers.
     */
    @Override
    synchronized
    public void log(String text, int stacktraceOffset) {
        super.log(text, stacktraceOffset);
        LogEntry logEntry = getLastLogEntry();
        logEntries.add(logEntry);
    }

    /** Getter für den Dateinamen. */
    @Override
    public String getLogFileName() {
        throw new RuntimeException("Der MemoryLogger schreibt in keine Datei!");
    }

    /** Gibt den Pfad zurück, in dem die Logdatei liegt. */
    @Override
    public String getLogPath() {
        throw new RuntimeException("Der MemoryLogger schreibt in keine Datei!");
    }

    /**
     * Gibt die zuletzt geschriebene Zeilennummer (oder anders gesagt die Anzahl an Zeilen im
     * Logger) zurück.
     *
     * @return Zeilennummer der zuletzt geschriebenen Zeile.
     */
    @Override
    public int getLastWrittenLineNumber() {
        return super.getLastWrittenLineNumber() + 1; // + 1 für die Titelzeile.
    }

    /** Getter für die geloggten Informationen. */
    public List<LogEntry> getLogEntries() {
        List<LogEntry> copiedLogEntries = new ArrayList<>();
        copiedLogEntries.addAll(logEntries);
        return copiedLogEntries;
    }

    /** Gibt eine Stringrepresentation für Debuggingzwecke zurück. */
    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("MemoryLogger\n");
        for (LogEntry logEntry : logEntries) {
            builder.append(logEntry.asLine()).append("\n");
        }
        return builder.toString();
    }

}
