package de.duehl.basics.io.textfile.dictionary.join;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

import de.duehl.basics.io.textfile.dictionary.Dictionary;
import de.duehl.basics.io.textfile.dictionary.DictionaryEntry;

public class DictionaryJoinerTest {

    @Test
    public void joinSimpleCase() {
        DictionaryEntry fooEntry = new DictionaryEntry("foo");
        fooEntry.addAlternative("bar");
        fooEntry.addAlternative("baz");

        Dictionary dictionary1 = new Dictionary();
        dictionary1.addDictionaryEntry(fooEntry);

        DictionaryEntry apfelEntry = new DictionaryEntry("Apfel");
        apfelEntry.addAlternative("Boskop");
        apfelEntry.addAlternative("Elstar");

        Dictionary dictionary2 = new Dictionary();
        dictionary2.addDictionaryEntry(apfelEntry);

        assertEquals(1, dictionary1.size());
        assertEquals(1, dictionary2.size());

        DictionaryJoiner joiner = new DictionaryJoiner(dictionary1, dictionary2);
        joiner.join();
        Dictionary dictionary = joiner.getJoinedDictionary();

        assertEquals(2, dictionary.size());

        assertTrue(dictionary.containsEntryWithMainWord("foo"));
        assertTrue(dictionary.containsEntryWithMainWord("Apfel"));

        DictionaryEntry actualFooEntry = dictionary.getEntryWithMainWord("foo");
        assertEquals(3, actualFooEntry.size());
        assertEquals("bar", actualFooEntry.getAlternatives().get(0));
        assertEquals("baz", actualFooEntry.getAlternatives().get(1));

        DictionaryEntry actualApfelEntry = dictionary.getEntryWithMainWord("Apfel");
        assertEquals(3, actualApfelEntry.size());
        assertEquals("Boskop", actualApfelEntry.getAlternatives().get(0));
        assertEquals("Elstar", actualApfelEntry.getAlternatives().get(1));
    }

    @Test
    public void joinComplexCase() {
        DictionaryEntry fooEntry1 = new DictionaryEntry("foo");
        fooEntry1.addAlternative("bar");

        DictionaryEntry apfelEntry1 = new DictionaryEntry("Apfel");
        apfelEntry1.addAlternative("Boskop");
        apfelEntry1.addAlternative("Elstar");

        DictionaryEntry tierEntry = new DictionaryEntry("Tier");
        tierEntry.addAlternative("Katze");
        tierEntry.addAlternative("Hund");
        tierEntry.addAlternative("Hase");

        Dictionary dictionary1 = new Dictionary();
        dictionary1.addDictionaryEntry(fooEntry1);
        dictionary1.addDictionaryEntry(apfelEntry1);
        dictionary1.addDictionaryEntry(tierEntry);


        DictionaryEntry fooEntry2 = new DictionaryEntry("foo");
        fooEntry2.addAlternative("bar");
        fooEntry2.addAlternative("baz");

        DictionaryEntry apfelEntry2 = new DictionaryEntry("Apfel");
        apfelEntry2.addAlternative("Elstar");

        DictionaryEntry baumEntry = new DictionaryEntry("Baum");
        baumEntry.addAlternative("Eiche");
        baumEntry.addAlternative("Buche");
        baumEntry.addAlternative("Ahorn");
        baumEntry.addAlternative("Erle");

        Dictionary dictionary2 = new Dictionary();
        dictionary2.addDictionaryEntry(fooEntry2);
        dictionary2.addDictionaryEntry(apfelEntry2);
        dictionary2.addDictionaryEntry(baumEntry);


        assertEquals(3, dictionary1.size());
        assertEquals(3, dictionary2.size());


        DictionaryJoiner joiner = new DictionaryJoiner(dictionary1, dictionary2);
        joiner.join();
        Dictionary dictionary = joiner.getJoinedDictionary();

        assertEquals(4, dictionary.size());

        assertTrue(dictionary.containsEntryWithMainWord("foo"));
        assertTrue(dictionary.containsEntryWithMainWord("Apfel"));
        assertTrue(dictionary.containsEntryWithMainWord("Tier"));
        assertTrue(dictionary.containsEntryWithMainWord("Baum"));

        DictionaryEntry actualFooEntry = dictionary.getEntryWithMainWord("foo");
        assertEquals(3, actualFooEntry.size());
        assertEquals("bar", actualFooEntry.getAlternatives().get(0));
        assertEquals("baz", actualFooEntry.getAlternatives().get(1));

        DictionaryEntry actualApfelEntry = dictionary.getEntryWithMainWord("Apfel");
        assertEquals(3, actualApfelEntry.size());
        assertEquals("Boskop", actualApfelEntry.getAlternatives().get(0));
        assertEquals("Elstar", actualApfelEntry.getAlternatives().get(1));

        DictionaryEntry actualTierEntry = dictionary.getEntryWithMainWord("Tier");
        assertEquals(4, actualTierEntry.size());
        assertEquals("Katze", actualTierEntry.getAlternatives().get(0));
        assertEquals("Hund", actualTierEntry.getAlternatives().get(1));
        assertEquals("Hase", actualTierEntry.getAlternatives().get(2));

        DictionaryEntry actualBaumEntry = dictionary.getEntryWithMainWord("Baum");
        assertEquals(5, actualBaumEntry.size());
        assertEquals("Eiche", actualBaumEntry.getAlternatives().get(0));
        assertEquals("Buche", actualBaumEntry.getAlternatives().get(1));
        assertEquals("Ahorn", actualBaumEntry.getAlternatives().get(2));
        assertEquals("Erle", actualBaumEntry.getAlternatives().get(3));
    }

}
