package de.duehl.swing.persistance;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;

import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;

/**
 * Diese Klasse stellt die grafische Oberfläche einer Farbe in einem Optionendialog dar.
 *
 * Hier wird diese Farben dargestellt und ggf. vom Benutzer verändert.
 *
 * @version 1.01     2025-07-25
 * @author Christian Dühl
 */

public class ColorOptionUi {

    /** Der Titel der beschreibt, welche Farbe hier dargestellt und bearbeitet wird. */
    private final String title;

    /** Die Farbe die hier dargestellt und bearbeitet wird. */
    private Color color;

    /** Der Panel auf dem die Farben dargestellt werden. */
    private final JPanel panel;

    /**
     * Das Label auf dem der Titel, der beschreibt, welche Farbe hier dargestellt und bearbeitet
     * wird, dargestellt wird.
     */
    private final JLabel titleLabel;

    /**
     * Konstruktor.
     *
     * @param title
     *            Der Titel der beschreibt, welche Farbe hier dargestellt und bearbeitet wird.
     */
    public ColorOptionUi(String title) {
        this(title, Color.BLACK);
    }

    /**
     * Konstruktor.
     *
     * @param title
     *            Der Titel der beschreibt, welche Farbe hier dargestellt und bearbeitet wird.
     * @param color
     *            Die Farbe die hier dargestellt und bearbeitet wird.
     */
    public ColorOptionUi(String title, Color color) {
        this.title = title;
        this.color = color;

        panel = new JPanel();
        titleLabel = new JLabel(title);

        init();
        createUi();
    }

    private void init() {
        GuiTools.biggerFont(titleLabel, 5);
        titleLabel.setOpaque(true);
        titleLabel.setBackground(Color.WHITE);
        titleLabel.setHorizontalAlignment(JLabel.CENTER);

        actualiseColors();
    }

    private void createUi() {
        //panel.removeAll();
        panel.setLayout(new BorderLayout());

        panel.add(titleLabel, BorderLayout.CENTER);
        panel.add(createColorChangeButton(), BorderLayout.EAST);
    }

    private Component createColorChangeButton() {
        JButton button = new JButton("Farbe ändern");
        button.addActionListener(e -> changeColor());
        return button;
    }

    private void changeColor() {
        Color newColor = JColorChooser.showDialog(panel,
                title + " festlegen",
                color);
        if (null != newColor) {
            setColor(newColor);
        }
    }

    /** Setter für die Farbe die hier dargestellt und bearbeitet wird. */
    public void setColor(Color color) {
        this.color = color;
        actualiseColors();
    }

    private void actualiseColors() {
        colorTitleLabel();
    }

    private void colorTitleLabel() {
        titleLabel.setForeground(color);
    }

    /** Getter für den Panel auf dem die Farbe dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

    /** Getter für die Farbe die hier dargestellt und bearbeitet wird. */
    public Color getColor() {
        return color;
    }

}
