package de.duehl.swing.text.html.characteristic;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
//import java.awt.GridLayout;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JColorChooser;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.ColorTranslator;
import de.duehl.swing.ui.layout.VerticalLayout;

/**
 * Diese Klasse erzeugt ein Panel für einen Optionendialog, um die Eigenschaften eines Stücks
 * Textes, welcher in einem HTML-Editor eingefärbt dargestellt wird, auswählen zu können.
 *
 * @version 1.01     2021-12-15
 * @author Christian Dühl
 */

public class OpticalTextPartCharacteristicPanel {

    private static final int COMBOBOX_PART_WIDTH = 330;

    /** Beschreibt welche Art von Textdarstellung hier eingestellt wird. */
    private final String description;

    /** Komponente des Optionen-Dialogs. */
    private final Component windowAsComponent;

    /**
     * Gibt an, ob die Beschreibung nur im Rahmen angezeigt wird (true) oder in jedem einzelnen
     * Einstellungselement (false).
     */
    private boolean useDescriptionOnlyInHeader;

    private final JButton foregroundColorChangeButton;
    private final JButton backgroundColorChangeButton;

    private Color foregroundColor;
    private Color backgroundColor;
    private final JCheckBox useBackgroundColorCheckBox;

    private final JCheckBox boldCheckBox;
    private final JCheckBox italicCheckBox;
    private final JCheckBox underlineCheckBox;

    /**
     * Konstruktor.
     *
     * @param description
     *            Beschreibt welche Art von Textdarstellung hier eingestellt wird.
     * @param windowAsComponent
     *            Komponente des Optionen-Dialogs.
     */
    public OpticalTextPartCharacteristicPanel(String description, Component windowAsComponent) {
        this.description = description;
        this.windowAsComponent = windowAsComponent;

        foregroundColorChangeButton = new JButton();
        backgroundColorChangeButton = new JButton();
        useBackgroundColorCheckBox = new JCheckBox();
        boldCheckBox = new JCheckBox();
        italicCheckBox = new JCheckBox();
        underlineCheckBox = new JCheckBox();

        useDescriptionOnlyInHeader = false;
    }

    /** Zeigt die Beschreibung nur im Rahmen an, nicht in jedem einzelnen Einstellungselement. */
    public void useDescriptionOnlyInHeader() {
        useDescriptionOnlyInHeader = true;
    }

    public void initElements() {
        setTitles();

        foregroundColorChangeButton.addActionListener(e -> selectForegroundColor());
        backgroundColorChangeButton.addActionListener(e -> selectBackgroundColor());
    }

    private void setTitles() {
        String front = determineTitleFront();

        foregroundColorChangeButton.setText(front + "Vordergrundfarbe ändern");
        backgroundColorChangeButton.setText(front + "Hintergrundfarbe ändern");
        useBackgroundColorCheckBox.setText(front + "Hintergrundfarbe nutzen");
        boldCheckBox.setText(front + "fett darstellen");
        italicCheckBox.setText(front + "kursiv darstellen");
        underlineCheckBox.setText(front + "unterstrichen darstellen");
    }

    private String determineTitleFront() {
        if (useDescriptionOnlyInHeader) {
            return "";
        }
        else {
            return description + " - ";
        }
    }

    private void selectForegroundColor() {
        Color color = JColorChooser.showDialog(windowAsComponent,
                description + " - Vordergrundfarbe im Text festlegen",
                foregroundColor);
        if (null != color) {
            foregroundColor = color;
            setButtonColors();
        }
    }

    private void selectBackgroundColor() {
        Color color = JColorChooser.showDialog(windowAsComponent,
                description + " - Hintergrundfarbe im Text festlegen",
                backgroundColor);
        if (null != color) {
            backgroundColor = color;
            setButtonColors();
        }
    }

    public Component createPanel() {
        JPanel panel = new JPanel();
        //panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        //panel.setLayout(new GridLayout(1, 2, 1, 1));
        panel.setLayout(new BorderLayout());
        //GuiTools.createTitle(description, panel);

        panel.add(createColorSelectionPart(), BorderLayout.CENTER);
        panel.add(createOptionsSelectionPart(), BorderLayout.EAST);

        return panel;
    }

    private Component createColorSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle(description + " - Farben", panel);

        panel.add(foregroundColorChangeButton);
        panel.add(backgroundColorChangeButton);
        panel.add(useBackgroundColorCheckBox);

        return panel;
    }

    private Component createOptionsSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(3, VerticalLayout.BOTH));
        GuiTools.createTitle(description + " - Textauszeichnung", panel);

        panel.add(boldCheckBox);
        panel.add(italicCheckBox);
        panel.add(underlineCheckBox);

        panel.setPreferredSize(new Dimension(COMBOBOX_PART_WIDTH, 1));

        return panel;
    }

    public void fillOptionValuesIntoGui(OpticalTextPartCharacteristic characteristics) {
        foregroundColor = ColorTranslator.hex2Swing(characteristics.getForegroundHexColor());
        backgroundColor = ColorTranslator.hex2Swing(characteristics.getBackgroundHexColor());

        useBackgroundColorCheckBox.setSelected(characteristics.isUseBackgroundColor());

        boldCheckBox.setSelected(characteristics.isBold());
        italicCheckBox.setSelected(characteristics.isItalic());
        underlineCheckBox.setSelected(characteristics.isUnderline());

        setButtonColors();
    }

    private void setButtonColors() {
        GuiTools.setForegroundAndBackground(foregroundColorChangeButton, foregroundColor);
        GuiTools.setForegroundAndBackground(backgroundColorChangeButton, backgroundColor);
    }

    /** Schreibt die Werte aus den Gui-Elementen in die übergebenen Darstellungseigenschaften. */
    public void apply(OpticalTextPartCharacteristic characteristics) {
        characteristics.setForegroundHexColor(ColorTranslator.swing2Hex(foregroundColor));
        characteristics.setBackgroundHexColor(ColorTranslator.swing2Hex(backgroundColor));

        characteristics.setUseBackgroundColor(useBackgroundColorCheckBox.isSelected());

        characteristics.setBold(boldCheckBox.isSelected());
        characteristics.setItalic(italicCheckBox.isSelected());
        characteristics.setUnderline(underlineCheckBox.isSelected());
    }

}
