package de.duehl.swing.ui.colors;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.util.ArrayList;
import java.util.List;

/**
 * Diese Klasse stellt eine Liste von benannten Farbe dar und bietet verschiedene Methoden,
 * benannte Farben oder Swing-Farben aus dieser Liste zu erhalten.
 *
 * @version 1.01     2020-01-16
 * @author Christian Dühl
 */

public class NamedColorList {

    /** Liste mit den benannten Farben. */
    private final List<NamedColor> namedColors;

    /** Konstruktor. */
    public NamedColorList() {
        namedColors = new ArrayList<>();
    }

    /**
     * Fügt eine Farbe zur Liste hinzu.
     *
     * @param namedColor
     *            Hinzuzufügende Farbe.
     * @throws IllegalArgumentException
     *             Falls bereits eine Farbe unter diesem Namen bekannt war.
     */
    public void add(NamedColor namedColor) {
        if (existsNamedColorByName(namedColor.getName())) {
            throw new IllegalArgumentException("Eine Farbe mit dem Namen '"
                    + namedColor.getName() + "' ist bereits vorhanden!");
        }
        namedColors.add(namedColor);
    }

    private boolean existsNamedColorByName(String name) {
        for (NamedColor namedColor : namedColors) {
            if (namedColor.getName().equals(name)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Getter für eine benannte Farbe über ihren Namen.
     *
     * @param name
     *            Name der Farbe.
     * @return Benannte Farbe.
     * @throws IllegalArgumentException
     *             falls keine Farbe unter diesem Namen bekannt war.
     */
    public NamedColor getNamedColorByName(String name) {
        for (NamedColor namedColor : namedColors) {
            if (namedColor.getName().equals(name)) {
                return namedColor;
            }
        }

        throw new IllegalArgumentException("Keine Farbe mit dem Namen '" + name + "' vorhanden!");
    }

    /**
     * Getter für eine benannte Farbe über ihre hexadezimale Stringdarstellung.
     *
     * @param hexValue
     *            Hexadezimale Stringdarstellung der Farbe.
     * @return Benannte Farbe.
     * @throws IllegalArgumentException
     *             falls keine Farbe mit dieser hexadezimalen Stringdarstellung bekannt war.
     */
    public NamedColor getNamedColorByHexcode(String hexValue) {
        for (NamedColor namedColor : namedColors) {
            if (namedColor.getHexValue().equals(hexValue)) {
                return namedColor;
            }
        }

        throw new IllegalArgumentException("Keine Farbe mit der hexadezimale Stringdarstellung '"
                + hexValue + "' vorhanden!");
    }

    /**
     * Getter für eine Swing-Farbe über ihren Namen.
     *
     * @param name
     *            Name der Farbe.
     * @return Swing-Farbe.
     * @throws IllegalArgumentException
     *             falls keine Farbe unter diesem Namen bekannt war.
     */
    public Color getColorByName(String name) {
        NamedColor namedColor = getNamedColorByName(name);
        return namedColor.getColor();
    }

    /**
     * Getter für eine Swing-Farbe über ihre hexadezimale Stringdarstellung.
     *
     * @param hexValue
     *            Hexadezimale Stringdarstellung der Farbe.
     * @return Swing-Farbe.
     * @throws IllegalArgumentException
     *             falls keine Farbe mit dieser hexadezimalen Stringdarstellung bekannt war.
     */
    public Color getColorByHexcode(String hexValue) {
        NamedColor namedColor = getNamedColorByHexcode(hexValue);
        return namedColor.getColor();
    }

    /**
     * Getter für die hexadezimale Stringdarstellung einer Farbe über ihren Namen.
     *
     * @param name
     *            Name der Farbe.
     * @return Hexadezimale Stringdarstellung.
     * @throws IllegalArgumentException
     *             falls keine Farbe unter diesem Namen bekannt war.
     */
    public String getHexCodeFromNamedColorByName(String name) {
        NamedColor namedColor = getNamedColorByName(name);
        return namedColor.getHexValue();
    }

}
