package de.duehl.swing.ui.dialogs.values;

/*
 * Copyright 2018 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JLabel;
import javax.swing.JTextField;

import de.duehl.swing.ui.colors.Colorizer;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zum Eingeben eines Wertes.
 *
 * @version 1.02     2018-02-08
 * @author Christian Dühl
 */

public class EnterValueDialog extends ModalDialogBase {

    private static final Dimension DIALOG_DIMENSION = new Dimension(500, 80);

    /** Feld in das der Benutzer seine Daten eingibt. */
    private final JTextField valueField;

    /** Text der Frage über dem Eingabefeld. */
    private final String question;

    /** Vorgabe die im Eingabefeld angezeigt werden soll. */
    private final String presetting;

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Icon für den Dialog.
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param question
     *            Text der Frage über dem Eingabefeld.
     * @param presetting
     *            Vorgabe die im Eingabefeld angezeigt werden soll.
     */
    public EnterValueDialog(Point parentLocation, Image programImage, String dialogTitle,
            String question, String presetting) {
        this(null, parentLocation, programImage, dialogTitle, question, presetting);
    }

    /**
     * Konstruktor.
     *
     * @param colorizer
     *            Farbverwaltung für die Gui.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Icon für den Dialog.
     * @param dialogTitle
     *            Titel des Dialogs.
     * @param question
     *            Text der Frage über dem Eingabefeld.
     * @param presetting
     *            Vorgabe die im Eingabefeld angezeigt werden soll.
     */
    public EnterValueDialog(Colorizer colorizer, Point parentLocation, Image programImage,
            String dialogTitle, String question, String presetting) {
        super(parentLocation, programImage, dialogTitle, DIALOG_DIMENSION, colorizer);
        this.question = question;
        this.presetting = presetting;

        valueField = new JTextField();

        fillDialog();

        addClosingWindowListener(() -> escape());
    }

    /** Erstellt die Dialog-Gui. */
    @Override
    protected void populateDialog() {
        add(createQuestionPart(), BorderLayout.CENTER);
        add(createEnterValuePart(), BorderLayout.SOUTH);
        setKeyBindings();
    }

    /** Erzeugt den Bereich mit der Frage. */
    private Component createQuestionPart() {
        JLabel label = new JLabel(question);
        setColors(label);

        return label;
    }

    /** Erzeugt den Bereich mit dem Textfeld. */
    private Component createEnterValuePart() {
        setColors(valueField);
        valueField.setCaretColor(Color.RED);
        valueField.addKeyListener(new KeyListener() {
            @Override
            public void keyPressed(KeyEvent arg0) {
            }
            @Override
            public void keyReleased(KeyEvent event) {
                if (KeyEvent.VK_ENTER == event.getKeyCode()) {
                    closeDialog();
                }
            }
            @Override
            public void keyTyped(KeyEvent arg0) {
            }
        });
        valueField.setText(presetting);
        valueField.selectAll();
        return valueField;
    }

    /** Erstellt Tastenkombinationen. */
    private void setKeyBindings() {
        setKeyBindingEscape(() -> escape());
    }

    private void escape() {
        valueField.setText(""); // Leer bei Abbruch
        closeDialog();
    }

    /** Getter für den Vom Benutzer eingegebener Text. */
    public String getEnteredValue() {
        return valueField.getText().trim();
    }

}
