package de.duehl.swing.ui.tabs.bars.generic;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Component;

import javax.swing.JTabbedPane;

import de.duehl.basics.text.NumberString;
import de.duehl.swing.ui.tabs.bars.growing.GrowingTabbedPane;
import de.duehl.swing.ui.tabs.bars.growing.TabCreator;
import de.duehl.swing.ui.tabs.elements.TabElements;

/**
 * Diese Klasse stellt ein Reiterelement dar, welches eine Reihe von benannten Komponenten
 * per Hinzufügen-Button hinzufügen kann.
 *
 * Dabei wird in jeden Reiter oben ein eigenes Tabulator-Objekt eingefügt, das anzeigen kann,
 * ob der Inhalt des Reiters geändert wurde oder unverändert ist.
 *
 * Die Klasse T steht für den Typ der Inhalte in den Reitern.
 *
 * @version 1.01     2021-11-30
 * @author Christian Dühl
 */

public class GrowingTabBar<T extends Component> extends TabBar<T> {

    /**
     * Konstruktor.
     *
     * @param tabCreator
     *            Objekt das einen neuen Reiter anlegen kann.
     */
    public GrowingTabBar(TabCreator tabCreator) {
        super(createTabs(tabCreator));
    }

    /**
     * Konstruktor.
     *
     * @param showCloseButton
     *            Gibt an, ob der Button zum Löschen angezeigt werden soll.
     * @param tabCreator
     *            Objekt das einen neuen Reiter anlegen kann.
     */
    public GrowingTabBar(boolean showCloseButton, TabCreator tabCreator) {
        super(showCloseButton, createTabs(tabCreator));
    }

    /** Erzeugt eine spezielle Reiterverwaltung für die wachsenden Reiter. */
    private static JTabbedPane createTabs(TabCreator tabCreator) {
        return new GrowingTabbedPane(tabCreator, JTabbedPane.TOP, JTabbedPane.SCROLL_TAB_LAYOUT);
    }

    /** Erzeugt aus dem gegebenen Titel einen noch nicht vergebenen Titel. */
    @Override
    public String generateNotUsedTitle(String title) {
        if (title.matches("new \\d+")) {
            return generateNotUsedTitleForNewTabs(title);
        }
        else {
            return super.generateNotUsedTitle(title);
        }
    }

    private String generateNotUsedTitleForNewTabs(String title) {
        String front = title.substring(0,  4);
        String numberString = title.substring(4);
        int number = NumberString.parseIntIgnore(numberString, 0);
        String notUsedTitle = front + Integer.toString(number);
        while (isTitleAlreadyInUse(notUsedTitle)) {
            notUsedTitle = front + Integer.toString(++number);
        }
        return notUsedTitle;
    }

    /**
     * Gibt die Tab-Elemente des aktuell angezeigten Reiters zurück.
     * Null falls nicht ermittelbar.
     */
    @Override
    public TabElements<T> getSelectedTabElements() {
        if (isPlusButtonTabulatorSelected()) {
            return null;
        }
        else {
            return super.getSelectedTabElements();
        }
    }

    private boolean isPlusButtonTabulatorSelected() {
        int index = getSelectedIndex();
        return isIndexOfPlusButtonTabulator(index);
    }

    private boolean isIndexOfPlusButtonTabulator(int index) {
        return index == getTabCount() - 1;
    }

}
