package de.duehl.swing.ui.start.search;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;

import java.awt.Dimension;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;
import de.duehl.swing.ui.elements.textfield.EnterReaktor;
import de.duehl.swing.ui.elements.textfield.EnterTextField;
import de.duehl.swing.ui.highlightingeditor.HighlightingEditor;
import de.duehl.swing.ui.highlightingeditor.HighlightingEditorWithoutButtonBar;

/**
 * Diese Klasse testet das EnterTextField-Element.
 *
 * @version 1.01     2017-12-15
 * @author Christian Dühl
 */

public class GotoDemoWithEditor {

    private final HighlightingEditor editor;

    private EnterTextField gotoField;

    /** Konstruktor. */
    public GotoDemoWithEditor() {
        gotoField = new EnterTextField(createEnterReaktor());

        editor = new HighlightingEditorWithoutButtonBar();
        editor.setText(Text.loremImpsum());

        SwingUtilities.invokeLater(() -> createAndShowGUI());
    }

    private EnterReaktor createEnterReaktor() {
        return new EnterReaktor() {
            @Override
            public void enterPressed(String text) {
                gotoLine(text);
            }
        };
    }

    private void gotoLine(String text) {
        if (!text.isEmpty() && NumberString.isDigitSequence(text)) {
            gotoLine(NumberString.parseIntIgnore(text, 0));
        }
    }

    private void gotoLine(int lineNumber) {
        if (lineNumber >= 1) {
            int startPositon = determineStartPositionOfSentence(lineNumber);
            editor.setCaretPosition(startPositon);
        }
    }

    private int determineStartPositionOfSentence(int lineNumber) {
        if (lineNumber == 1) {
            return 0;
        }

        int lineNumberZeroBased = lineNumber - 1;
        String text = getTextWithInternalLineBreaks();

        int index = 0;
        int found = 0;
        int lastIndex;

        while (index != -1 && found < lineNumberZeroBased) {
            lastIndex = index;
            index = text.indexOf("\n", index) + "\n".length();
            if (index < lastIndex) {
                return text.length();
            }
            ++found;
        }

        return index;
    }

    private String getTextWithInternalLineBreaks() {
        String text = editor.getText();
        text = Text.lineBreaksToBackslashN(text);
        return text;
    }

    private void createAndShowGUI() {
        JFrame frame = new JFrame("SearchPanel Test");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setLayout(new BorderLayout());

        frame.add(gotoField, BorderLayout.NORTH);
        frame.add(editor.getComponent(), BorderLayout.CENTER);

        frame.setPreferredSize(new Dimension(800, 600));
        frame.pack();
        frame.setVisible(true);
    }

    /** Startpunkt des Programms, Parameter werden ignoriert. */
    public static void main(String[] args) {
        new GotoDemoWithEditor();
    }

}

