package de.duehl.vocabulary.japanese.logic.success;

import java.awt.Color;
import java.util.List;

import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalKanjiData;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;

/**
 * Diese Klasse ermittelt den Erfolg bei den letzten zehn Abfragen eines Liste von Kanji und
 * erzeugt daraus die passende Farbe sowie die Prozentzahl.
 *
 * @version 1.01     2024-11-07
 * @author Christian Dühl
 */

public class KanjiListTestSuccesssCalculator {

    /** Die Liste mit den Kanji, zu welchen der Erfolg berechnet werden soll. */
    private final List<Kanji> kanjiList;

    /** Die Programmoptionen. */
    private final Options options;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalKanjiDataRequester requester;

    /** Die Anzahl der maximal zehn Abfragen aller Kanji des Vokabulars. */
    private int lastCountSum;

    /** Die Anzahl der korrekten maximal zehn Abfragen aller Kanji des Vokabulars. */
    private int lastCorrectCountSum;

    /** Die Vordergrundfarbe für den Namen des Vokabulars in der VocabularyBar. */
    private Color foregroundColor;

    /** Der Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar. */
    private String percentText;

    /** Die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    private double percent;

    /**
     * Der Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur Anzeige in der
     * VocabularyBar.
     */
    private String percentTextWithTwoDigitsAfterComma;

    /**
     * Konstruktor.
     *
     * @param kanjiList
     *            Die Liste mit den Kanji,  zu welchen der Erfolg berechnet werden soll.
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     */
    public KanjiListTestSuccesssCalculator(List<Kanji> kanjiList, Options options,
            InternalKanjiDataRequester requester) {
        this.kanjiList = kanjiList;
        this.options = options;
        this.requester = requester;
    }

    /** Führt die Berechnung durch. */
    public void calculate() {
        calculateCounts();
        foregroundColor = determineForegroundColor();
        determinePercentAndPercentText();
    }

    private void calculateCounts() {
        lastCountSum = 0;
        lastCorrectCountSum = 0;

        for (Kanji kanji : kanjiList) {
            calculateCountsForKanji(kanji);
        }
    }

    private void calculateCountsForKanji(Kanji kanji) {
        InternalAdditionalKanjiData data = requester.getInternalDataForKanji(kanji);
        int lastCount;
        int lastCorrectCount;
        lastCount = data.getLastTenTestsCount();
        lastCorrectCount = data.getLastCorrectTestsCount();
        lastCountSum += lastCount;
        lastCorrectCountSum += lastCorrectCount;

        /* Damit nicht abgefragte Kanji nicht gut aussehen: */
        if (lastCount == 0) {
            lastCountSum += 10;
        }
    }

    private Color determineForegroundColor() {
        if (options.isColorVocabularyDependingOnLastSuccess()) {
            VocableColors vocableColors = new VocableColors(options);
            return vocableColors.determineForegroundColor(lastCountSum, lastCorrectCountSum);
        }
        else {
            return Color.BLACK;
        }
    }

    private void determinePercentAndPercentText() {
        percentTextWithTwoDigitsAfterComma = NumberString.percent(lastCorrectCountSum, lastCountSum);

        percent = NumberString.percentAsNumber(lastCorrectCountSum, lastCountSum);

        String percentString = String.format("%.0f", percent);
        percentText = Text.fillWithSpacesAtFront(percentString, 3);
    }

    /**
     * Getter für die Vordergrundfarbe.
     *
     * Bei der entsprechenden Klasse für die Berechnung des Erfolgs für Vokabeln wird diese Methide
     * dazu benutzt, den Namen des Vokabulars in der VocabularyBar einzufärben. Ob es hier bei den
     * Kanji für diese Methode eine Verwendung gibt, muss man sehen.
     */
    public Color getForegroundColor() {
        return foregroundColor;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen zur Anzeige in der VocabularyBar.
     */
    public String getPercentText() {
        return percentText;
    }

    /**
     * Getter für den Text zum Erfolg der letzten zehn Abfragen mit zwei Nachkommastellen zur
     * Anzeige in der VocabularyBar.
     */
    public String getPercentTextWithTwoDigitsAfterComma() {
        return percentTextWithTwoDigitsAfterComma;
    }

    /** Getter für die Prozentzahl des Erfolgs der letzten zehn Abfragen. */
    public double getPercent() {
        return percent;
    }

}
