package de.duehl.vocabulary.japanese.ui.components.button;

import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Point;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.logic.view.CompleteVocabularyViewerLogic;
import de.duehl.vocabulary.japanese.ui.data.MessageSetter;
import de.duehl.vocabulary.japanese.ui.dialog.VocabularyListerDialog;
import de.duehl.vocabulary.japanese.ui.dialog.VocabularySheetDialog;

/**
 * Diese Klasse zeigt drei Buttons an, mit denen man eine Menge von Vokabeln in Form einer
 * Einzeldarstellung, in Form einer Liste und in Form eines Vokabelblattes anzeigen lassen kann.
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

public class OtherViewButtonPanel {

    /** Die Liste mit den anzuzeigenden Vokabeln. */
    private final List<Vocable> vocables;

    /**
     * Der OtherViewButtonPanel anzeigende Dialog, der die eigene Anzeige beendet, wenn in dieser
     * Klasse eine andere Anzeige aufgerufen wurde.
     */
    private final ViewButtonPressedReactor reactor;

    /** Die Beschreibung der Menge von Vokabeln. */
    private final String description;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalDataRequester requester;

    /** Die Programmoptionen. */
    private final Options options;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalKanjiDataRequester kanjiRequester;

    /** Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen kann. */
    private final LongTimeProcessInformer informer;

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /** Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann. */
    private final MessageSetter messageSetter;

    /** Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird. */
    private final Point parentLocation;

    /** Das anzuzeigende ProgrammIcon. */
    private final Image programImage;

    /** Der Panel mit den drei Buttons. */
    private final JPanel panel;

    /** Der Button zur Ansicht der Einzeldarstellung. */
    private final JButton einzeldarstellungButton;

    /** Der Button zur Ansicht als Liste. */
    private final JButton listenDarstellungButton;

    /** Der Button zur Ansicht als Vokabelblatt. */
    private final JButton blattDarstellungButton;

    /**
     * Konstruktor.
     *
     * @param vocables
     *            Die Liste mit den anzuzeigenden Vokabeln.
     * @param reactor
     *            Der OtherViewButtonPanel anzeigende Dialog, der die eigene Anzeige beendet, wenn
     *            in dieser Klasse eine andere Anzeige aufgerufen wurde.
     * @param description
     *            Die Beschreibung der Menge von Vokabeln.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param options
     *            Die Programmoptionen.
     * @param kanjiRequester
     *            Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param informer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     * @param ownLists
     *            Die Verwaltung der eigenen Vokabellisten.
     * @param messageSetter
     *            Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann.
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das anzuzeigende ProgrammIcon.
     */
    public OtherViewButtonPanel(List<Vocable> vocables, ViewButtonPressedReactor reactor,
            String description, InternalDataRequester requester, Options options,
            InternalKanjiDataRequester kanjiRequester, LongTimeProcessInformer informer,
            OwnLists ownLists, MessageSetter messageSetter, Point parentLocation,
            Image programImage) {
        this.vocables = vocables;
        this.reactor = reactor;
        this.description = description;
        this.requester = requester;
        this.options = options;
        this.kanjiRequester = kanjiRequester;
        this.informer = informer;
        this.ownLists = ownLists;
        this.messageSetter = messageSetter;
        this.parentLocation = parentLocation;
        this.programImage = programImage;

        panel = new JPanel();
        einzeldarstellungButton = new JButton();
        listenDarstellungButton = new JButton();
        blattDarstellungButton = new JButton();

        createGui();
    }

    private void createGui() {
        initPanel();
        initEinzeldarstellungButton();
        initListenDarstellungButton();
        initVokabelblattDarstellungButton();

        fillPanel();
    }

    private void initPanel() {
        panel.setLayout(new GridLayout(0, 3, 5, 5));
    }

    private void initEinzeldarstellungButton() {
        einzeldarstellungButton.setText("Einzeldarstellung");
        einzeldarstellungButton.addActionListener(e -> einzeldarstellung());
    }

    private void einzeldarstellung() {
        closeOldViewer();

        CompleteVocabularyViewerLogic viewer = new CompleteVocabularyViewerLogic(options, requester,
                vocables, kanjiRequester, description, informer, ownLists, messageSetter,
                parentLocation, programImage);
        viewer.view();
    }

    private void closeOldViewer() {
        reactor.otherViewOpened();
    }

    private void initListenDarstellungButton() {
        listenDarstellungButton.setText("Liste");
        listenDarstellungButton.addActionListener(e -> listenDarstellung());
    }

    private void listenDarstellung() {
        closeOldViewer();

        VocabularyListerDialog dialog = new VocabularyListerDialog(options, requester, vocables,
                description, kanjiRequester, informer, ownLists, messageSetter, parentLocation,
                programImage);
        dialog.setVisible(true);
    }

    private void initVokabelblattDarstellungButton() {
        blattDarstellungButton.setText("Blatt");
        blattDarstellungButton.addActionListener(e -> vokabelblattDarstellung());
    }

    private void vokabelblattDarstellung() {
        closeOldViewer();
        informer.startLongTimeProcess("Öffne Blattdarstellung mit " + vocables.size() + " Vokabeln");
        new Thread(() -> listVocabularyAsSheetInOwnThread()).start();
    }

    private void listVocabularyAsSheetInOwnThread() {
        VocabularySheetDialog dialog = new VocabularySheetDialog(options, requester, vocables,
                description, kanjiRequester, informer, ownLists, messageSetter, parentLocation,
                programImage);
        listVocabularyAsSheetInEdt(dialog);
    }

    private void listVocabularyAsSheetInEdt(VocabularySheetDialog dialog) {
        dialog.setVisible(true);
        dialog.requestFocus();
        informer.endLongTimeProcess();
    }

    private void fillPanel() {
        panel.add(einzeldarstellungButton);
        panel.add(listenDarstellungButton);
        panel.add(blattDarstellungButton);
    }

    /** Deaktiviert den Button zur Ansicht der Einzeldarstellung. */
    public void disableEinzeldarstellungButton() {
        einzeldarstellungButton.setEnabled(false);
    }

    /** Deaktiviert den Button zur Ansicht als Liste. */
    public void disableListenDarstellungButton() {
        listenDarstellungButton.setEnabled(false);
    }

    /** Deaktiviert den Button zur Ansicht als Vokabelblatt. */
    public void disableVokabelblattDarstellungButton() {
        blattDarstellungButton.setEnabled(false);
    }

    /** Getter für den Panel mit den drei Buttons. */
    public JPanel getPanel() {
        return panel;
    }

}
