package de.duehl.vocabulary.japanese.ui.dialog.kana.hiraganatest.components;

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.JPanel;

import de.duehl.swing.ui.components.selections.SelectionsHelper;
import de.duehl.swing.ui.components.selections.StringSelection;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalKanaData;
import de.duehl.vocabulary.japanese.data.symbol.Hiragana;
import de.duehl.vocabulary.japanese.logic.symbol.kana.internal.InternalKanaDataRequester;

/**
 * Diese Klasse stellt ein einzelnes Hiragana Panel dar.
 *
 * Gezeigt werden Das Zeichen selbst, die deutsche Bedeutung, die ON-Lesungen und die kun-Lesungen.
 *
 * @version 1.01     2025-02-02
 * @author Christian Dühl
 */

public class HiraganaPanel {

    /** Das Label auf dem das Hiragana dargestellt wird. */
    private final HiraganaLabel hiraganaLabel;

    /** Gibt an, ob auch die internen Daten zum Hiragana angezeigt werden sollen. */
    private boolean showInternalHiraganaData;

    /**
     * Gibt an, ob das Zeichen auch nochmal in einem Viewer angezeigt werden soll, damit es
     * heraus kopiert werden kann.
     */
    private boolean showHiraganaAsViewerToo;

    /** Das Objekt, das zu einem Hiragana die internen, benutzerabhängigen Daten abrufen kann. */
    private InternalKanaDataRequester requester;

    /** Der Panel auf dem das Hiragana dargestellt wird. */
    private final JPanel panel;

    private final StringSelection hiraganaAsCharacterSeletion;
    private final StringSelection hepburnSelection;
    private final StringSelection subTypeSelection;

    private final StringSelection testCountSelection;
    private final StringSelection correctTestCountSelection;
    private final StringSelection lastTestDatetSelection;
    private final StringSelection lastCorrectTestDateSelection;
    private final StringSelection lastTenTestResultsSelection;

    /** Konstruktor. */
    public HiraganaPanel() {
        hiraganaLabel = new HiraganaLabel();
        hiraganaLabel.setBiggerFontSize(200); // Default: 400

        showInternalHiraganaData = false;
        showHiraganaAsViewerToo = false;


        panel = new JPanel();

        hiraganaAsCharacterSeletion = new StringSelection("Das Hiragana (noch einmal zum rauskopieren)");
        hepburnSelection = new StringSelection("Hepburn-Darstellung");
        subTypeSelection = new StringSelection("Unterart");

        testCountSelection = new StringSelection("Anzahl Test");
        correctTestCountSelection = new StringSelection("Anzahl erfolgreicher Test");
        lastTestDatetSelection = new StringSelection("zuletzt getestet am");
        lastCorrectTestDateSelection = new StringSelection("zuletzt erfolgreich getestet am");
        lastTenTestResultsSelection = new StringSelection("Erfolg der letzten Tests");

        init();
    }

    private void init() {
        initAssetFocusableViewer(hiraganaAsCharacterSeletion);
        initAssetFocusableViewer(hiraganaAsCharacterSeletion);
        initAssetFocusableViewer(hepburnSelection);
        initAssetFocusableViewer(subTypeSelection);

        initAssetFocusableViewer(testCountSelection);
        initAssetFocusableViewer(correctTestCountSelection);
        initAssetFocusableViewer(lastTestDatetSelection);
        initAssetFocusableViewer(lastCorrectTestDateSelection);
        initAssetFocusableViewer(lastTenTestResultsSelection);
    }

    private void initAssetFocusableViewer(StringSelection selection) {
        SelectionsHelper.initSelectionAsViewer(selection);
        selection.setFocusable(true);
    }

    public void showHiraganaAsViewerToo() {
        showHiraganaAsViewerToo = true;
    }

    /**
     * Legt fest, dass auch die internen Daten zum Hiragana angezeigt werden sollen.
     *
     * @param requester
     *            Das Objekt, das zu einem Hiragana die internen, benutzerabhängigen Daten abrufen
     *            kann.
     */
    public void showInternalHiraganaDataToo(InternalKanaDataRequester requester) {
        showInternalHiraganaData = true;
        this.requester = requester;
    }

    /**
     * Erzeugt die Gui.
     *
     * Möchte man die internen Daten anzeigen, so muss man zuvor die Methode
     * showInternalHiraganaDataToo(requester) aufrufen!
     */
    public void createGui() {
        panel.setLayout(new BorderLayout());
        //GuiTools.createTitle(panel); Das sollte man wohl außerhalb entscheiden.

        panel.add(hiraganaLabel.getLabel(), BorderLayout.CENTER);
        panel.add(createGermanMeaningOnLesungenAndKunLesungenAndPerhapsInternalDataPart(),
                BorderLayout.SOUTH);
    }

    private Component createGermanMeaningOnLesungenAndKunLesungenAndPerhapsInternalDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        if (showHiraganaAsViewerToo) {
            panel.add(hiraganaAsCharacterSeletion.getPanel());
        }
        panel.add(createHepburnAndSubTypePart());
        if (showInternalHiraganaData) {
            panel.add(createInternalHiraganaDataPart());
        }
        return panel;
    }

    private Component createHepburnAndSubTypePart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(hepburnSelection.getPanel());
        panel.add(subTypeSelection.getPanel());

        return panel;
    }

    private Component createInternalHiraganaDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(testCountSelection.getPanel());
        panel.add(correctTestCountSelection.getPanel());
        panel.add(lastTestDatetSelection.getPanel());
        panel.add(lastCorrectTestDateSelection.getPanel());
        panel.add(lastTenTestResultsSelection.getPanel());

        return panel;
    }

    /** Getter für den Panel auf dem das Hiragana dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

    /** Zeigt das Hiragana an. Kann mehrfach aufgerufen werden. */
    public void showHiragana(Hiragana hiragana) {
        hiraganaLabel.showHiragana(hiragana);

        hiraganaAsCharacterSeletion.setText(hiragana.getCharacter());
        hepburnSelection.setText(hiragana.getHepburn());
        subTypeSelection.setText(hiragana.getSubType().getDescription());

        if (showInternalHiraganaData) {
            InternalAdditionalKanaData data = requester.getInternalDataForHiragana(hiragana);

            testCountSelection.setText(data.getTestCount());
            correctTestCountSelection.setText(data.getCorrectTestCount());
            lastTestDatetSelection.setText(data.getLastTestDate().toString());
            lastCorrectTestDateSelection.setText(data.getLastCorrectTestDate().toString());
            lastTenTestResultsSelection.setText(data.getLastTenTestResultsAsStorageString());
        }
    }

}
