package de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjitest;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Point;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;

import de.duehl.basics.text.NumberString;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.KanjiSet;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.test.data.KanjiForTestSelectionMethod;

/**
 * Diese Klasse lässt den Benutzer die zugrundeliegende Menge von Kanji auswählen, außerdem ob
 * einige oder alle Kanji und ob die Kanji in der originalen Reihenfolge oder zufällig sortiert
 * abgefragt werden sollen.
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

public class KanjiTestParameterChooser extends ModalDialogBase {

    private static final int MIN_WIDTH = 500;


    /** Die Liste der benutzerdefinierten Kanji-Mengen. */
    private final List<KanjiSet> kanjiSets;

    /** Die Programmoptionen. */
    private final Options options;

    /** Die Liste der nicht zuletzt zehn mal richtig abgefragten Kanji. */
    private final List<Kanji> notTenTimesGoodTestedKanjiList;

    /** Die Combobox für die Kanji-Menge, die man abfragen will. */
    private final JComboBox<String> kanjiSetSelectionComboBox;

    /**
     * Das Verzeichnis der Listen von Kanji nach dem Text in der ComboBox zur Auswahl der
     * Kanji-Menge.
     */
    private final Map<String, List<Kanji>> kanjiListByComboBoxText;

    /** Gibt an, ob der Benutzer den Dialog mit OK beendet hat. */
    private boolean applied;

    /* Gui-Elemente: */

    private final JCheckBox useOriginalKanjiOrderCheckBox;

    private final JRadioButton fiveKanjiRadioButton;
    private final JRadioButton tenKanjiRadioButton;
    private final JRadioButton fifteenKanjiRadioButton;
    private final JRadioButton twenytKanjiRadioButton;
    private final JRadioButton thirtyKanjiRadioButton;
    private final JRadioButton fiftyKanjiRadioButton;
    private final JRadioButton allKanjiRadioButton;
    private final JRadioButton notTenTimesGoodTestedKanjiRadioButton;
    private final JRadioButton lastFiftyKanjiRadioButton;
    private final JRadioButton lastThirtyKanjiRadioButton;
    private final JRadioButton lastTwentyKanjiRadioButton;
    private final JRadioButton lastFifteenKanjiRadioButton;
    private final JRadioButton lastTenKanjiRadioButton;
    private final JRadioButton lastFiveKanjiRadioButton;
    private final JRadioButton oldestTestedFiftyKanjiRadioButton;
    private final JRadioButton oldestTestedThirtyKanjiRadioButton;
    private final JRadioButton oldestTestedTwentyKanjiRadioButton;
    private final JRadioButton oldestTestedFifteenKanjiRadioButton;
    private final JRadioButton oldestTestedTenKanjiRadioButton;
    private final JRadioButton oldestTestedFiveKanjiRadioButton;
    private final JRadioButton mostSeldomTestedFiftyKanjiRadioButton;
    private final JRadioButton mostSeldomTestedThirtyKanjiRadioButton;
    private final JRadioButton mostSeldomTestedTwentyKanjiRadioButton;
    private final JRadioButton mostSeldomTestedFifteenKanjiRadioButton;
    private final JRadioButton mostSeldomTestedTenKanjiRadioButton;
    private final JRadioButton mostSeldomTestedFiveKanjiRadioButton;

    private final JCheckBox germanMeaningCaseSensitivityCheckBox;
    private final JCheckBox onLesungCaseSensitivityCheckBox;
    private final JCheckBox kunLesungCaseSensitivityCheckBox;

    /** Der Button zum positiven Abschicken des Dialogs. */
    private final JButton okButton;

    /** Die Anzahl der abzufragenden Kanji. */
    private int numberOfKanjiToTest;

    /** Gibt an ob die abzufragenden Kanji in der originalen Reihenfolge belassen werden sollen. */
    private boolean useOriginalKanjiOrder;

    /** Die Art wie Kanji für die Abfrage ausgewählt werden. */
    private KanjiForTestSelectionMethod selectionMethod;

    /**
     * Gibt an, ob bei der Überprüfung der deutschen Bedeutung auf Groß-/Kleinschreibung geachtet
     * werden soll.
     */
    private boolean germanMeaningCaseSensitivity;

    /**
     * Gibt an, ob bei der Überprüfung der ON-Lesung auf Groß-/Kleinschreibung geachtet werden
     * soll.
     */
    private boolean onLesungCaseSensitivity;

    /**
     * Gibt an, ob bei der Überprüfung der kun-Lesung auf Groß-/Kleinschreibung geachtet werden
     * soll.
     */
    private boolean kunLesungCaseSensitivity;

    /** Die Liste an Kanji, aus denen ausgewählt wird. */
    private List<Kanji> kanjiList;

    /**
     * Konstruktor.
     *
     * @param kanjiSets
     *            Die Liste der benutzerdefinierten Kanji-Mengen.
     * @param options
     *            Die Programmoptionen.
     * @param notTenTimesGoodTestedKanjiList
     *            Die Liste der nicht zuletzt zehn mal richtig abgefragten Kanji.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Anzuzeigendes ProgrammIcon.
     */
    public KanjiTestParameterChooser(List<KanjiSet> kanjiSets, Options options,
            List<Kanji> notTenTimesGoodTestedKanjiList, Point parentLocation, Image programImage) {
        super(parentLocation, programImage, "Die Parameter für den Kanji-Test auswählen");

        this.kanjiSets = kanjiSets;
        this.options = options;
        this.notTenTimesGoodTestedKanjiList = notTenTimesGoodTestedKanjiList;

        kanjiSetSelectionComboBox = new JComboBox<>();
        kanjiListByComboBoxText = new HashMap<>();

        useOriginalKanjiOrderCheckBox = new JCheckBox("in Originalreihenfolge");

        fiveKanjiRadioButton = new JRadioButton("5 Kanji");
        tenKanjiRadioButton = new JRadioButton("10 Kanji");
        fifteenKanjiRadioButton = new JRadioButton("15 Kanji");
        twenytKanjiRadioButton = new JRadioButton("20 Kanji");
        thirtyKanjiRadioButton = new JRadioButton("30 Kanji");
        fiftyKanjiRadioButton = new JRadioButton("50 Kanji");
        allKanjiRadioButton = new JRadioButton("Alle Kanji ("
                + NumberString.taupu(Kanji.getNumberOfKnownKanji()) + ")");
        notTenTimesGoodTestedKanjiRadioButton = new JRadioButton(
                "nicht zuletzt zehn mal richtig abgefragte Kanji");
        lastFiftyKanjiRadioButton = new JRadioButton("die letzten 50 Kanji");
        lastThirtyKanjiRadioButton = new JRadioButton("die letzten 30 Kanji");
        lastTwentyKanjiRadioButton = new JRadioButton("die letzten 20 Kanji");
        lastFifteenKanjiRadioButton = new JRadioButton("die letzten 15 Kanji");
        lastTenKanjiRadioButton = new JRadioButton("die letzten 10 Kanji");
        lastFiveKanjiRadioButton = new JRadioButton("die letzten 5 Kanji");
        oldestTestedFiftyKanjiRadioButton = new JRadioButton(
                "die 50 am längsten nicht mehr abgefragten Kanji");
        oldestTestedThirtyKanjiRadioButton = new JRadioButton(
                "die 30 am längsten nicht mehr abgefragten Kanji");
        oldestTestedTwentyKanjiRadioButton = new JRadioButton(
                "die 20 am längsten nicht mehr abgefragten Kanji");
        oldestTestedFifteenKanjiRadioButton = new JRadioButton(
                "die 15 am längsten nicht mehr abgefragten Kanji");
        oldestTestedTenKanjiRadioButton = new JRadioButton(
                "die 10 am längsten nicht mehr abgefragten Kanji");
        oldestTestedFiveKanjiRadioButton = new JRadioButton(
                "die 5 am längsten nicht mehr abgefragten Kanji");
        mostSeldomTestedFiftyKanjiRadioButton = new JRadioButton(
                "die 50 am seltensten abgefragten Kanji");
        mostSeldomTestedThirtyKanjiRadioButton = new JRadioButton(
                "die 30 am seltensten abgefragten Kanji");
        mostSeldomTestedTwentyKanjiRadioButton = new JRadioButton(
                "die 20 am seltensten abgefragten Kanji");
        mostSeldomTestedFifteenKanjiRadioButton = new JRadioButton(
                "die 15 am seltensten abgefragten Kanji");
        mostSeldomTestedTenKanjiRadioButton = new JRadioButton(
                "die 10 am seltensten abgefragten Kanji");
        mostSeldomTestedFiveKanjiRadioButton = new JRadioButton(
                "die 5 am seltensten abgefragten Kanji");

        germanMeaningCaseSensitivityCheckBox = new JCheckBox("Bei der Überprüfung der deutschen "
                + "Bedeutung auf Groß-/Kleinschreibung achten.");
        onLesungCaseSensitivityCheckBox = new JCheckBox("Bei der Überprüfung der ON-Lesung auf "
                + "Groß-/Kleinschreibung achten.");
        kunLesungCaseSensitivityCheckBox = new JCheckBox("Bei der Überprüfung der kun-Lesung auf "
                + "Groß-/Kleinschreibung achten.");

        okButton = new JButton();

        selectionMethod = KanjiForTestSelectionMethod.UNKNOWN;
        applied = false;

        init();
        fillDialog();
    }

    private void init() {
        initComboBox();
        fillComboBox();
        addComboBoxActionListener();
        initRadioButtons();
        initCheckBoxes();
    }

    private void initComboBox() {
        kanjiSetSelectionComboBox.setAlignmentX(JLabel.CENTER);
        kanjiSetSelectionComboBox.setAlignmentY(JLabel.CENTER);
        ((JLabel) kanjiSetSelectionComboBox.getRenderer()).setHorizontalAlignment(
                SwingConstants.CENTER);
    }

    private void fillComboBox() {
        kanjiSetSelectionComboBox.removeAllItems();

        String allKanjiText = "Alle bekannten Kanji";
        List<Kanji> allKanji = Kanji.getAllKanjiAsList();
        kanjiListByComboBoxText.put(allKanjiText, allKanji);
        kanjiSetSelectionComboBox.addItem(allKanjiText);


        for (KanjiSet kanjiSet : kanjiSets) {
            String name = kanjiSet.getName();
            String text = "Menge '" + name + "'";
            List<Kanji> kanjiList = kanjiSet.getSet();
            kanjiListByComboBoxText.put(text, kanjiList);
            kanjiSetSelectionComboBox.addItem(text);
        }

        String lastText = options.getLastSelectedKanjiSetTextInCombobox();
        if (kanjiListByComboBoxText.containsKey(lastText)) {
            kanjiSetSelectionComboBox.setSelectedItem(lastText);
        }
        else {
            kanjiSetSelectionComboBox.setSelectedItem(allKanjiText);
        }

        comboboxSelectionChanged();
    }

    private void addComboBoxActionListener() {
        kanjiSetSelectionComboBox.addActionListener(e -> comboboxSelectionChanged());
    }

    private void comboboxSelectionChanged() {
        String text = (String)  kanjiSetSelectionComboBox.getSelectedItem();

        if (!kanjiListByComboBoxText.containsKey(text)) {
            quit();
            throw new RuntimeException("Der Text aus der Combobox '" + text
                    + "' ist im Verzeichnis der Kanjilisten nach dem Text nicht bekannt.");
        }

        List<Kanji> selectedKanjiList = kanjiListByComboBoxText.get(text);

        int numberOfAllKanji = selectedKanjiList.size();
        allKanjiRadioButton.setText("Alle Kanji (" + NumberString.taupu(numberOfAllKanji) + ")");

        int numberOfNotTenTimesGoodTestedKanji = notTenTimesGoodTestedKanjiList.size();
        notTenTimesGoodTestedKanjiRadioButton.setText(
                "nicht zuletzt zehn mal richtig abgefragte Kanji" + " ("
                        + numberOfNotTenTimesGoodTestedKanji + ")");
        notTenTimesGoodTestedKanjiRadioButton.setEnabled(!notTenTimesGoodTestedKanjiList.isEmpty());
        // Berücksichtigt noch nicht ob die Kanji zur Menge passen!
        // Das ist aber Absicht.

        handleRadioButton(fiveKanjiRadioButton, 5, numberOfAllKanji);
        handleRadioButton(tenKanjiRadioButton, 10, numberOfAllKanji);
        handleRadioButton(fifteenKanjiRadioButton, 15, numberOfAllKanji);
        handleRadioButton(twenytKanjiRadioButton, 20, numberOfAllKanji);
        handleRadioButton(thirtyKanjiRadioButton, 30, numberOfAllKanji);
        handleRadioButton(fiftyKanjiRadioButton, 50, numberOfAllKanji);
        handleRadioButton(lastFiftyKanjiRadioButton, 50, numberOfAllKanji);
        handleRadioButton(lastThirtyKanjiRadioButton, 30, numberOfAllKanji);
        handleRadioButton(lastTwentyKanjiRadioButton, 20, numberOfAllKanji);
        handleRadioButton(lastFifteenKanjiRadioButton, 15, numberOfAllKanji);
        handleRadioButton(lastTenKanjiRadioButton, 10, numberOfAllKanji);
        handleRadioButton(lastFiveKanjiRadioButton, 5, numberOfAllKanji);
        handleRadioButton(oldestTestedFiftyKanjiRadioButton, 50, numberOfAllKanji);
        handleRadioButton(oldestTestedThirtyKanjiRadioButton, 30, numberOfAllKanji);
        handleRadioButton(oldestTestedTwentyKanjiRadioButton, 20, numberOfAllKanji);
        handleRadioButton(oldestTestedFifteenKanjiRadioButton, 15, numberOfAllKanji);
        handleRadioButton(oldestTestedTenKanjiRadioButton, 10, numberOfAllKanji);
        handleRadioButton(oldestTestedFiveKanjiRadioButton, 5, numberOfAllKanji);
        handleRadioButton(mostSeldomTestedFiftyKanjiRadioButton, 50, numberOfAllKanji);
        handleRadioButton(mostSeldomTestedThirtyKanjiRadioButton, 30, numberOfAllKanji);
        handleRadioButton(mostSeldomTestedTwentyKanjiRadioButton, 20, numberOfAllKanji);
        handleRadioButton(mostSeldomTestedFifteenKanjiRadioButton, 15, numberOfAllKanji);
        handleRadioButton(mostSeldomTestedTenKanjiRadioButton, 10, numberOfAllKanji);
        handleRadioButton(mostSeldomTestedFiveKanjiRadioButton, 5, numberOfAllKanji);
    }

    private void handleRadioButton(JRadioButton nKanjiRadioButton, int numberOnRadioButton,
            int numberOfAllKanji) {
        nKanjiRadioButton.setEnabled(numberOfAllKanji >= numberOnRadioButton);
        if (numberOfAllKanji < numberOnRadioButton && nKanjiRadioButton.isSelected()) {
            nKanjiRadioButton.setSelected(false);
        }
    }

    private void initRadioButtons() {
        ButtonGroup group = new ButtonGroup();

        group.add(fiveKanjiRadioButton);
        group.add(tenKanjiRadioButton);
        group.add(fifteenKanjiRadioButton);
        group.add(twenytKanjiRadioButton);
        group.add(thirtyKanjiRadioButton);
        group.add(fiftyKanjiRadioButton);
        group.add(allKanjiRadioButton);
        group.add(notTenTimesGoodTestedKanjiRadioButton);
        group.add(lastFiftyKanjiRadioButton);
        group.add(lastThirtyKanjiRadioButton);
        group.add(lastTwentyKanjiRadioButton);
        group.add(lastFifteenKanjiRadioButton);
        group.add(lastTenKanjiRadioButton);
        group.add(lastFiveKanjiRadioButton);
        group.add(oldestTestedFiftyKanjiRadioButton);
        group.add(oldestTestedThirtyKanjiRadioButton);
        group.add(oldestTestedTwentyKanjiRadioButton);
        group.add(oldestTestedFifteenKanjiRadioButton);
        group.add(oldestTestedTenKanjiRadioButton);
        group.add(oldestTestedFiveKanjiRadioButton);
        group.add(mostSeldomTestedFiftyKanjiRadioButton);
        group.add(mostSeldomTestedThirtyKanjiRadioButton);
        group.add(mostSeldomTestedTwentyKanjiRadioButton);
        group.add(mostSeldomTestedFifteenKanjiRadioButton);
        group.add(mostSeldomTestedTenKanjiRadioButton);
        group.add(mostSeldomTestedFiveKanjiRadioButton);

        allKanjiRadioButton.setSelected(true);
    }

    private void initCheckBoxes() {
        useOriginalKanjiOrderCheckBox.setSelected(false);

        germanMeaningCaseSensitivityCheckBox.setSelected(false);
        onLesungCaseSensitivityCheckBox.setSelected(false);
        kunLesungCaseSensitivityCheckBox.setSelected(false);
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createCenterPart(), BorderLayout.CENTER);
        add(createButtonsPart(),  BorderLayout.SOUTH);

        SwingUtilities.invokeLater(() -> okButton.requestFocus());
    }

    private Component createCenterPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createKanjiSetSelectionPart(), BorderLayout.NORTH);
        panel.add(createNumberOfKanjiPart(), BorderLayout.CENTER);
        panel.add(createSortKanjiAndCaseSensitivityPart(), BorderLayout.SOUTH);

        return panel;
    }

    private Component createKanjiSetSelectionPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle("Teilmenge der abzufragenden Kanji", panel);

        panel.add(kanjiSetSelectionComboBox, BorderLayout.CENTER);

        return panel;
    }

    private Component createNumberOfKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));
        //panel.setLayout(new GridLayout(1, 0, 0, 0));

        panel.add(createNumberOfRandomSelectedKanjiPart());
        panel.add(createNumberOfLastXKanjiPart());
        panel.add(createNumberOfOldestTestedKanjiPart());
        panel.add(createNumberOfMostSeldomTestedKanjiPart());
        panel.add(createNumberOfAllKanjiPart());
        panel.add(createNumberOfNotTenTimesGoodTestedKanjiPart());
        panel.add(createDummyMinWidthLabelPart());

        return panel;
    }

    private Component createNumberOfRandomSelectedKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 3, 5, 5));
        GuiTools.createTitle("Für die Abfrage der Kanji die folgende Anzahl zufällig auswälhen",
                panel);

        panel.add(fiveKanjiRadioButton);
        panel.add(tenKanjiRadioButton);
        panel.add(fifteenKanjiRadioButton);
        panel.add(twenytKanjiRadioButton);
        panel.add(thirtyKanjiRadioButton);
        panel.add(fiftyKanjiRadioButton);

        return panel;
    }

    private Component createNumberOfLastXKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 3, 5, 5));
        GuiTools.createTitle("Für die Abfrage der Kanji die folgende Anzahl der zuletzt "
                + "hinzugefügten auswälhen", panel);

        panel.add(lastFiftyKanjiRadioButton);
        panel.add(lastThirtyKanjiRadioButton);
        panel.add(lastTwentyKanjiRadioButton);
        panel.add(lastFifteenKanjiRadioButton);
        panel.add(lastTenKanjiRadioButton);
        panel.add(lastFiveKanjiRadioButton);

        return panel;
    }

    private Component createNumberOfOldestTestedKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 3, 5, 5));
        GuiTools.createTitle("Für die Abfrage der Kanji die folgende Anzahl der am längsten nicht "
                + "mehr abgefragten Kanji auswälhen", panel);

        panel.add(oldestTestedFiftyKanjiRadioButton);
        panel.add(oldestTestedThirtyKanjiRadioButton);
        panel.add(oldestTestedTwentyKanjiRadioButton);
        panel.add(oldestTestedFifteenKanjiRadioButton);
        panel.add(oldestTestedTenKanjiRadioButton);
        panel.add(oldestTestedFiveKanjiRadioButton);

        return panel;
    }

    private Component createNumberOfMostSeldomTestedKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 3, 5, 5));
        GuiTools.createTitle("Für die Abfrage der Kanji die folgende Anzahl der am seltensten "
                + "abgefragten Kanji auswälhen", panel);

        panel.add(mostSeldomTestedFiftyKanjiRadioButton);
        panel.add(mostSeldomTestedThirtyKanjiRadioButton);
        panel.add(mostSeldomTestedTwentyKanjiRadioButton);
        panel.add(mostSeldomTestedFifteenKanjiRadioButton);
        panel.add(mostSeldomTestedTenKanjiRadioButton);
        panel.add(mostSeldomTestedFiveKanjiRadioButton);

        return panel;
    }

    private Component createNumberOfAllKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 3, 5, 5));
        GuiTools.createTitle("Alle Kanji abfragen", panel);

        panel.add(allKanjiRadioButton);

        return panel;
    }

    private Component createNumberOfNotTenTimesGoodTestedKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 3, 5, 5));
        GuiTools.createTitle("Nicht zehn mal richtig beantwortete Kanji abfragen", panel);

        panel.add(notTenTimesGoodTestedKanjiRadioButton);

        return panel;
    }

    private Component createDummyMinWidthLabelPart() {
        JLabel label = new JLabel("");
        label.setPreferredSize(new Dimension(MIN_WIDTH, 0));
        return label;
    }

    private Component createSortKanjiAndCaseSensitivityPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(createSortKanjiPart());
        panel.add(createCaseSensitivityPart());

        return panel;
    }

    private Component createSortKanjiPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle("Sortierung", panel);

        panel.add(useOriginalKanjiOrderCheckBox, BorderLayout.CENTER);

        return panel;
    }

    private Component createCaseSensitivityPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));
        GuiTools.createTitle("Groß-/Kleinschreibung", panel);

        panel.add(germanMeaningCaseSensitivityCheckBox);
        panel.add(onLesungCaseSensitivityCheckBox);
        panel.add(kunLesungCaseSensitivityCheckBox);

        return panel;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.WEST);
        panel.add(createOkButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createQuitButton() {
        JButton button = new JButton("Abbrechen");
        button.addActionListener(e -> quit());
        return button;
    }

    private void quit() {
        closeDialog();
    }

    private Component createOkButton() {
        okButton.setText("   Abfrage starten   ");
        GuiTools.biggerFont(okButton, 5);
        okButton.addActionListener(e -> apply());
        return okButton;
    }

    private void apply() {
        String text = (String)  kanjiSetSelectionComboBox.getSelectedItem();
        if (kanjiListByComboBoxText.containsKey(text)) {
            kanjiList = kanjiListByComboBoxText.get(text);
            options.setLastSelectedKanjiSetTextInCombobox(text);
        }
        else {
            quit();
            throw new RuntimeException("Der Text aus der Combobox '" + text
                    + "' ist im Verzeichnis der Kanjilisten nach dem Text nicht bekannt.");
        }

        if (fiveKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 5;
            selectionMethod = KanjiForTestSelectionMethod.RANDOM_BY_NUMBER;
        }
        else if (tenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 10;
            selectionMethod = KanjiForTestSelectionMethod.RANDOM_BY_NUMBER;
        }
        else if (fifteenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 15;
            selectionMethod = KanjiForTestSelectionMethod.RANDOM_BY_NUMBER;
        }
        else if (twenytKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 20;
            selectionMethod = KanjiForTestSelectionMethod.RANDOM_BY_NUMBER;
        }
        else if (thirtyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 30;
            selectionMethod = KanjiForTestSelectionMethod.RANDOM_BY_NUMBER;
        }
        else if (fiftyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 50;
            selectionMethod = KanjiForTestSelectionMethod.RANDOM_BY_NUMBER;
        }
        else if (allKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = Kanji.getNumberOfKnownKanji(); // eigentlich unnötig
            selectionMethod = KanjiForTestSelectionMethod.ALL;
        }
        else if (notTenTimesGoodTestedKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = notTenTimesGoodTestedKanjiList.size();
            selectionMethod = KanjiForTestSelectionMethod.NOT_TEN_TIMES_GOOD_TESTED;
        }
        else if (lastFiftyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 50;
            selectionMethod = KanjiForTestSelectionMethod.LAST_N;
        }
        else if (lastThirtyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 30;
            selectionMethod = KanjiForTestSelectionMethod.LAST_N;
        }
        else if (lastTwentyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 20;
            selectionMethod = KanjiForTestSelectionMethod.LAST_N;
        }
        else if (lastFifteenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 15;
            selectionMethod = KanjiForTestSelectionMethod.LAST_N;
        }
        else if (lastTenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 10;
            selectionMethod = KanjiForTestSelectionMethod.LAST_N;
        }
        else if (lastFiveKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 5;
            selectionMethod = KanjiForTestSelectionMethod.LAST_N;
        }
        else if (oldestTestedFiftyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 50;
            selectionMethod = KanjiForTestSelectionMethod.OLDEST_TESTED_N;
        }
        else if (oldestTestedThirtyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 30;
            selectionMethod = KanjiForTestSelectionMethod.OLDEST_TESTED_N;
        }
        else if (oldestTestedTwentyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 20;
            selectionMethod = KanjiForTestSelectionMethod.OLDEST_TESTED_N;
        }
        else if (oldestTestedFifteenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 15;
            selectionMethod = KanjiForTestSelectionMethod.OLDEST_TESTED_N;
        }
        else if (oldestTestedTenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 10;
            selectionMethod = KanjiForTestSelectionMethod.OLDEST_TESTED_N;
        }
        else if (oldestTestedFiveKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 5;
            selectionMethod = KanjiForTestSelectionMethod.OLDEST_TESTED_N;
        }
        else if (mostSeldomTestedFiftyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 50;
            selectionMethod = KanjiForTestSelectionMethod.MOST_SELDOM_TESTED_N;
        }
        else if (mostSeldomTestedThirtyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 30;
            selectionMethod = KanjiForTestSelectionMethod.MOST_SELDOM_TESTED_N;
        }
        else if (mostSeldomTestedTwentyKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 20;
            selectionMethod = KanjiForTestSelectionMethod.MOST_SELDOM_TESTED_N;
        }
        else if (mostSeldomTestedFifteenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 15;
            selectionMethod = KanjiForTestSelectionMethod.MOST_SELDOM_TESTED_N;
        }
        else if (mostSeldomTestedTenKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 10;
            selectionMethod = KanjiForTestSelectionMethod.MOST_SELDOM_TESTED_N;
        }
        else if (mostSeldomTestedFiveKanjiRadioButton.isSelected()) {
            numberOfKanjiToTest = 5;
            selectionMethod = KanjiForTestSelectionMethod.MOST_SELDOM_TESTED_N;
        }
        else {
            numberOfKanjiToTest = -1;
            selectionMethod = KanjiForTestSelectionMethod.UNKNOWN;

            String title = "Keine Anzahl an Kanji ausgewählt";
            String message = "Sie haben die Anzahl der abzufragenden Kanji nicht ausgewählt.";
            GuiTools.informUser(getWindowAsComponent(), title, message);
            return;
        }

        useOriginalKanjiOrder = useOriginalKanjiOrderCheckBox.isSelected();
        germanMeaningCaseSensitivity = germanMeaningCaseSensitivityCheckBox.isSelected();
        onLesungCaseSensitivity = onLesungCaseSensitivityCheckBox.isSelected();
        kunLesungCaseSensitivity = kunLesungCaseSensitivityCheckBox.isSelected();


        applied = true;
        quit();
    }

    /** Gibt an, ob der Benutzer den Dialog mit OK beendet hat. */
    public boolean isApplied() {
        return applied;
    }

    /** Die Getter für die an Kanji, aus denen ausgewählt wird. */
    public List<Kanji> getKanjiList() {
        return kanjiList;
    }

    /** Getter für die Anzahl der abzufragenden Kanji. */
    public int getNumberOfKanjiToTest() {
        return numberOfKanjiToTest;
    }

    /** Gibt an ob die abzufragenden Kanji in der originalen Reihenfolge belassen werden sollen. */
    public boolean isUseOriginalKanjiOrder() {
        return useOriginalKanjiOrder;
    }

    /** Getter für die Art wie Kanji für die Abfrage ausgewählt werden. */
    public KanjiForTestSelectionMethod getSelectionMethod() {
        return selectionMethod;
    }

    /**
     * Gibt an, ob bei der Überprüfung der deutschen Bedeutung auf Groß-/Kleinschreibung geachtet
     * werden soll.
     */
    public boolean isGermanMeaningCaseSensitivity() {
        return germanMeaningCaseSensitivity;
    }

    /**
     * Gibt an, ob bei der Überprüfung der ON-Lesung auf Groß-/Kleinschreibung geachtet werden
     * soll.
     */
    public boolean isOnLesungCaseSensitivity() {
        return onLesungCaseSensitivity;
    }

    /**
     * Gibt an, ob bei der Überprüfung der kun-Lesung auf Groß-/Kleinschreibung geachtet werden
     * soll.
     */
    public boolean isKunLesungCaseSensitivity() {
        return kunLesungCaseSensitivity;
    }

}
