package de.duehl.vocabulary.japanese.ui.dialog.options.colors.eleven;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.GridLayout;

import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.color.VocableColors;

/**
 * Diese Klasse stellt die grafische Oberfläche zum Anzeigen und Bearbeiten einer Farbe des
 * Vokabel-Trainers dar.
 *
 * Es gibt elf Farben, diese beziehen sich auf die elf Möglichkeiten, wie man die letzten zehn
 * Abfragen beantwortet haben kann:
 *
 *      1. Keine Abfrage richtig beantwortet, zehn Abfragen falsch beantwortet.
 *      2. Eine Abfrage richtig beantwortet, neun Abfragen falsch beantwortet.
 *      3. Zwei Abfragen richtig beantwortet, acht Abfragen falsch beantwortet.
 *      4. Drei Abfragen richtig beantwortet, sieben Abfragen falsch beantwortet.
 *      5. Vier Abfragen richtig beantwortet, sechs Abfragen falsch beantwortet.
 *      6. Fünf Abfragen richtig beantwortet, fünf Abfragen falsch beantwortet.
 *      7. Sechs Abfragen richtig beantwortet, vier Abfragen falsch beantwortet.
 *      8. Sieben Abfragen richtig beantwortet, drei Abfragen falsch beantwortet.
 *      9. Acht Abfragen richtig beantwortet, zwei Abfragen falsch beantwortet.
 *     10. Neun Abfragen richtig beantwortet, eine Abfrage falsch beantwortet.
 *     11. Zehn Abfragen richtig beantwortet, keine Abfrage falsch beantwortet.
 *
 * Die Farben gehen von rot (schlecht) über gelb (mittel) bis grün (gut).
 *
 * Hier wird eine dieser Farben dargestellt und ggf. vom Benutzer verändert.
 *
 * @version 1.01     2024-08-15
 * @author Christian Dühl
 */

public class OneOfElevenColorUi {

    /** Der Titel der beschreibt, welche Farbe hier dargestellt und bearbeitet wird. */
    private final String title;

    /** Die Farbe die hier dargestellt und bearbeitet wird. */
    private Color color;

    /** Erzeugt die passenden Farben. */
    private final VocableColors vocableColors;

    /** Der Panel auf dem die Farben dargestellt werden. */
    private final JPanel panel;

    /**
     * Das Label auf dem der Titel, der beschreibt, welche Farbe hier dargestellt und bearbeitet
     * wird, dargestellt wird.
     */
    private final JLabel titleLabel;

    /** Das Label auf dem exemplarisch die erzeugte Vordergrundfarbe dargestellt wird. */
    private final JLabel foregroundExampleLabel;

    /** Das Panel auf dem exemplarisch die erzeugte Hintergrundfarbe dargestellt wird. */
    private final JPanel backgroundExamplePanel;

    /**
     * Konstruktor.
     *
     * @param title
     *            Der Titel der beschreibt, welche Farbe hier dargestellt und bearbeitet wird.
     * @param color
     *            Die Farbe die hier dargestellt und bearbeitet wird.
     * @param vocableColors
     *            Erzeugt die passenden Farben.
     */
    public OneOfElevenColorUi(String title, Color color, VocableColors vocableColors) {
        this.title = title;
        this.color = color;
        this.vocableColors = vocableColors;

        panel = new JPanel();
        titleLabel = new JLabel(title);
        foregroundExampleLabel = new JLabel("  Vordergrund  ");
        backgroundExamplePanel = new JPanel();

        init();
        createUi();
    }

    private void init() {
        GuiTools.biggerFont(titleLabel, 5);
        titleLabel.setOpaque(true);
        titleLabel.setBackground(Color.WHITE);
        titleLabel.setHorizontalAlignment(JLabel.CENTER);

        GuiTools.biggerFont(foregroundExampleLabel, 5);
        actualiseColors();
    }

    private void createUi() {
        //panel.removeAll();
        panel.setLayout(new BorderLayout());

        panel.add(titleLabel, BorderLayout.CENTER);
        panel.add(createColorExamplePartAndChangeButton(), BorderLayout.EAST);
    }

    private Component createColorExamplePartAndChangeButton() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createColorExamplePart(), BorderLayout.CENTER);
        panel.add(createColorChangeButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createColorExamplePart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 2, 2, 2));

        panel.add(foregroundExampleLabel);
        panel.add(backgroundExamplePanel);

        return panel;
    }

    private Component createColorChangeButton() {
        JButton button = new JButton("Farbe ändern");
        button.addActionListener(e -> changeColor());
        return button;
    }

    private void changeColor() {
        Color newColor = JColorChooser.showDialog(panel,
                title + " festlegen",
                color);
        if (null != newColor) {
            color = newColor;
            actualiseColors();
        }
    }

    private void colorTitleLabel() {
        titleLabel.setForeground(color);
    }

    private void colorForegroundExampleLabel() {
        foregroundExampleLabel.setForeground(vocableColors.createForegroundColor(color));
    }

    private void colorBackgroundExamplePanel() {
        backgroundExamplePanel.setBackground(vocableColors.createBackgroundColor(color));
    }

    private void actualiseColors() {
        colorTitleLabel();
        colorForegroundExampleLabel();
        colorBackgroundExamplePanel();
    }

    /** Getter für den Panel auf dem die Farbe dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

    /** Getter für die Farbe die hier dargestellt und bearbeitet wird. */
    public Color getColor() {
        return color;
    }

}
