package de.duehl.basics.io.textfile.dictionary.io.reader;

import de.duehl.basics.caller.CallerDeterminer;
import de.duehl.basics.io.Charset;
import de.duehl.basics.io.textfile.StringsFromTextFileInJarReader;

/**
 * Diese Klasse liest eine Liste mit den Texten im Format des DictionaryFiles aus den Zeilen einer
 * Textdatei in einer Jar ein.
 *
 * Das Format sieht wie folgt aus:
 *
 * Englisches Wort
 *     Deutsches Wort 1
 *     Deutsches Wort 2
 *     Deutsches Wort 3
 *     Deutsches Wort 4
 *
 * Kommt in einer Datenzeile weiter hinten das Kommentarzeichen vor, so wird ab dort abgeschnitten
 * und getrimmt.
 *
 * @version 1.01     2020-01-28
 * @author Christian Dühl
 */

public class DictionaryFromTextFileInJarReader extends DictionaryFromTextFileReader {

    /** Kommentarzeichen '#' für Konstruktoren ohne diese Angabe. */
    private static final String DEFAULT_COMMENT_CHARACTER = "#";

    /**
     * Konstruktor für UTF-8 Files mit Standard-Kommentarzeichen '#' und der Klasse des Aufrufers,
     * neben der also die Textdatei erwartet wird.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     */
    public DictionaryFromTextFileInJarReader(String filename) {
        this(filename, Charset.UTF_8, CallerDeterminer.determineCallingClass(),
                DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor für UTF-8 Files mit Standard-Kommentarzeichen '#'.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     * @param clazz
     *            Klasse neben der die Datei sich im befindet.
     */
    public DictionaryFromTextFileInJarReader(String filename, Class<?> clazz) {
        this(filename, Charset.UTF_8, clazz, DEFAULT_COMMENT_CHARACTER);
    }

    /**
     * Konstruktor.
     *
     * @param filename
     *            Name der einzulesenden Datei ohne Pfad.
     * @param charset
     *            Kodierung der einzulesenden Datei.
     * @param clazz
     *            Klasse neben der die Datei sich im befindet.
     * @param commentChar
     *            Das Kommentarzeichen.
     */
    public DictionaryFromTextFileInJarReader(String filename, Charset charset, Class<?> clazz,
            String commentChar) {
        super(new StringsFromTextFileInJarReader(filename, charset, clazz, commentChar));
    }

}
