package de.duehl.swing.debug;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import de.duehl.basics.io.Charset;
import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.Text;
import de.duehl.swing.debug.data.DebugMessage;

import static de.duehl.swing.debug.GuiDebug.ALL_CATEGORY;

/**
 * Diese Klasse speichert die Debugausgaben in Dateien ab.
 *
 * @version 1.01     2025-07-30
 * @author Christian Dühl
 */

public class GuiDebugSaver {

    /** Die Kategorien zu denen Meldungen erfolgen. */
    private final List<String> categories;

    /** Die eingegangenen Meldungen nach Kategorie. */
    private final Map<String, List<DebugMessage>> messagesByCategory;

    /**
     * Konstruktor.
     *
     * @param categories
     *            Die Kategorien zu denen Meldungen erfolgen.
     * @param messagesByCategory
     *            Die eingegangenen Meldungen nach Kategorie.
      */
    public GuiDebugSaver(List<String> categories,
            Map<String, List<DebugMessage>> messagesByCategory) {
        this.categories = categories;
        this.messagesByCategory = messagesByCategory;
    }

    /** Speichert alle Meldungen in eine chronologische Datei. */
    public void saveAll(String filename) {
        List<String> lines = new ArrayList<>();
        appendToLines(lines, ALL_CATEGORY);
        writeFile(lines, filename);
    }

    /** Speichert die Meldungen nach Kategorien in eine Datei mit Blöcken pro Kategorie. */
    public void saveByCategories(String filename) {
        List<String> lines = new ArrayList<>();
        for (String category : categories) {
            appendToLines(lines, category);
        }

        writeFile(lines, filename);
    }

    private void appendToLines(List<String> lines, String category) {
        String intro;
        if (category.equals(ALL_CATEGORY)) {
            intro = "Alles:";
        }
        else {
            intro = category + ":";
        }
        lines.add(intro);
        String underline = Text.multipleString("~", intro.length());
        lines.add(underline);
        lines.add("");

        for (DebugMessage message : messagesByCategory.get(category)) {
            StringBuilder builder = new StringBuilder();
            builder.append(message.getDate().toString()).append(" ");
            builder.append(message.getTime().toString()).append(" ");
            if (category.equals(ALL_CATEGORY)) {
                builder.append(message.getCategory()).append(" ");
            }
            builder.append(message.getClassName()).append(" ");
            builder.append(message.getMethodName()).append(" ");
            builder.append(message.getLineNumber()).append(" ");
            builder.append(message.getMessage()).append(" ");
            String line = builder.toString();
            lines.add(line);
        }

        lines.add("");
        lines.add("");
    }

    private void writeFile(List<String> lines, String filename) {
        FileHelper.writeLinesToFile(lines, filename, Charset.UTF_8);
    }

}
