package de.duehl.swing.ui.buttons.panel;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.border.Border;
import javax.swing.border.EtchedBorder;

/**
 * Diese Klasse stellt einen JPanel als Button dar.
 *
 * Nach Code von https://stackoverflow.com/questions/39955390/how-to-use-a-jpanel-as-jbutton
 *
 * Der Button sollte eigentlich im VocableAfterTypingErrorSelectorDialog verwendet werden, leider
 * ließen die darauf angezeigten Objekte den Klick nicht durch, so dass ich dort eine andere Lösung
 * gewählt habe. Eingebaut hatte ich ihn dort so:
 *
 *       VocableViewer viewer = new VocableViewer(options);
 *       viewer.showVocable(vocable, requester.getInternalDataForVocable(vocable));
 *       Component viewerPanel = viewer.getPanel();
 *
 *       JPanelButton panelButton = new JPanelButton(() -> userSelectsVocable(vocable));
 *       panelButton.setLayout(new BorderLayout());
 *       panelButton.add(viewerPanel, BorderLayout.CENTER);
 *
 * @version 1.01     2024-03-27
 * @author Christian Dühl
 */

public class JPanelButton extends JPanel  {

    private static final long serialVersionUID = 1L;

    private static final Border RAISED_BORDER = BorderFactory.createEtchedBorder(
            EtchedBorder.RAISED);
    private static final Border LOWERED_BORDER = BorderFactory.createEtchedBorder(
            EtchedBorder.LOWERED);

    /** Das Objekt das aufgerufen wird, wenn der Benutzer den Button klickt. */
    private final PanelButtonClickReactor clickReactor;

    /**
     * Konstruktor.
     *
     * @param clickReactor
     *            Das Objekt das aufgerufen wird, wenn der Benutzer den Button klickt.
     */
    public JPanelButton(PanelButtonClickReactor clickReactor) {
        this.clickReactor = clickReactor;

        raiseBorder();
        setFocusable(true); // you need this or the panel won't get the key events
        addMouseListener(createMouseListener());
    }

    private MouseAdapter createMouseListener() {
        return new MouseAdapter() {
            @Override
            public void mousePressed(final MouseEvent e) {
                lowerBorder();

            }
            @Override
            public void mouseReleased(final MouseEvent e) {
                raiseBorder();
                clickReactor.userHasClicked();
            }
        };
    }

    private void raiseBorder() {
        setBorder(RAISED_BORDER);
    }

    private void lowerBorder() {
        setBorder(LOWERED_BORDER);
    }

}