package de.duehl.swing.ui.dialogs;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.Colorizer;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zur Anzeige des übergebenen Panels. Der Benutzer kann mit Ja
 * oder nein antworten. Abbrechen des Dialogs gilt als nein.
 *
 * @version 1.02     2019-05-24
 * @author Christian Dühl
 */

public class PanelDialogYesNo extends ModalDialogBase {

    /** Der Panel, der hier angezeigt wird. */
    private final JPanel panel;

    /** Antwort des Benutzers. */
    private boolean answeredYes;

    private final JButton yesButton;
    private final JButton noButton;

    /**
     * Konstruktor.
     *
     * @param panel
     *            Anzuzeigender Panel.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog
     *            erzeugt wird.
     * @param programImage
     *            Image für den Dialog.
     * @param title
     *            Titel des Dialogs.
     */
    public PanelDialogYesNo(JPanel panel, Point parentLocation, Image programImage, String title) {
        this(panel, null, parentLocation, programImage, title);
    }

    /**
     * Konstruktor.
     *
     * @param panel
     *            Anzuzeigender Panel.
     * @param colorizer
     *            Farbverwaltung für die Gui.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog
     *            erzeugt wird.
     * @param programImage
     *            Image für den Dialog.
     * @param title
     *            Titel des Dialogs.
     */
    public PanelDialogYesNo(JPanel panel, Colorizer colorizer, Point parentLocation,
            Image programImage, String title) {
        super(parentLocation, programImage, title, colorizer);
        addClosingWindowListener(() -> no());

        this.panel = panel;
        answeredYes = false;
        yesButton = createYesButton();
        noButton = createNoButton();

        fillDialog();
    }

    private JButton createYesButton() {
        JButton button = new JButton("Ja");
        setColors(button);
        GuiTools.boldFont(button);
        button.requestFocusInWindow();
        button.addActionListener(e -> yes());

        return button;
    }

    private JButton createNoButton() {
        JButton button = new JButton("Nein");
        setColors(button);
        button.addActionListener(e -> no());

        return button;
    }

    @Override
    protected void populateDialog() {
        add(panel, BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);

        addEscapeBehaviour();
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        setColors(panel);
        panel.setLayout(new BorderLayout());

        panel.add(createYesNoButtons(), BorderLayout.EAST);

        return panel;
    }

    private Component createYesNoButtons() {
        JPanel panel = new JPanel();
        setColors(panel);
        panel.setLayout(new BorderLayout());

        panel.add(yesButton, BorderLayout.WEST);
        panel.add(noButton, BorderLayout.EAST);

        return panel;
    }

    private void yes() {
        answeredYes = true;
        closeDialog();
    }

    private void no() {
        answeredYes = false;
        closeDialog();
    }

    public boolean isAnsweredYes() {
        return answeredYes;
    }

    public void setYesTextTo(String yesText) {
        yesButton.setText(yesText);
    }

    public void setNoTextTo(String noText) {
        noButton.setText(noText);
    }

}
