package de.duehl.swing.ui.dialogs;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Image;
import java.awt.Point;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zur Eingabe eines Passworts.
 *
 * @version 1.02     2024-12-10
 * @author Christian Dühl
 */

public class PasswordDialog extends ModalDialogBase {

    /** Der Text des Dialogs. */
    private final String message;

    private final JPasswordField passwordField;

    /**
     * Das eingegebene Passwort. Ost es leer, wurde die Bearbeitung abgebrochen oder das Passwort
     * leer eingegeben.
     */
    private String password;

    /**
     * Konstruktor.
     *
     * @param title
     *            Der Titel des Dialogs.
     * @param message
     *            Der Text des Dialogs.
     */
    public PasswordDialog(String title, String message) {
        this(new Point(150, 100), null, title, message);
    }

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog
     *            erzeugt wird.
     * @param programImage
     *            Das Image für den Dialog.
     * @param title
     *            Der Titel des Dialogs.
     * @param message
     *            Der Text des Dialogs.
     */
    public PasswordDialog(Point parentLocation, Image programImage, String title, String message) {
        super(parentLocation, programImage, title);
        this.message = message;

        passwordField = new JPasswordField(16);

        init();
        fillDialog();
    }

    private void init() {
        password = "";
        GuiTools.biggerFont(passwordField, 10);
        GuiTools.addReturnListener(passwordField, () -> ok());
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createMessagePart(), BorderLayout.NORTH);
        add(passwordField, BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);
    }

    private Component createMessagePart() {
        JLabel label = new JLabel(message);
        GuiTools.biggerFont(label, 5);
        return label;
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        setColors(panel);
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.WEST);
        panel.add(createOKButton(), BorderLayout.EAST);

        return panel;
    }

    private JButton createQuitButton() {
        JButton button = new JButton("Abbrechen");
        setColors(button);
        button.addActionListener(e -> quit());

        return button;
    }

    private void quit() {
        closeDialog();
    }

    private Component createOKButton() {
        JButton button = new JButton("OK");
        setColors(button);
        button.addActionListener(e -> ok());

        return button;
    }

    private void ok() {
        char[] pass = passwordField.getPassword();
        password = new String(pass).strip();
        closeDialog();
    }

    /**
     * Getter für das eingegebene Passwort. Ost es leer, wurde die Bearbeitung abgebrochen oder das
     * Passwort leer eingegeben.
     */
    public String getPassword() {
        return password;
    }

}
