package de.duehl.swing.ui.error;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.lang.reflect.InvocationTargetException;

import javax.swing.SwingUtilities;

/**
 * Diese Klasse registriert einen UncaughtExceptionHandler, welcher Fehler abfängt und mittels
 * eines <tt>ErrorDialog</tt>s anzeigt.
 *
 * @version 1.04     2022-01-27
 * @author Christian Dühl
 */

public class GuiExceptionHandlerRegistration {

    /** Der ExceptionHandler, der hier registriert wird. */
    private final GuiExceptionHandler exceptionHandler;

    /** Konstruktor. */
    public GuiExceptionHandlerRegistration() {
        exceptionHandler = new GuiExceptionHandler();
    }

    /** Setter für die Aktion nach Anzeige des Fehlers. */
    public void setAfterErrorHandlingCallable(AfterErrorHandlingCallable callable) {
        exceptionHandler.setAfterErrorHandlingCallable(callable);
    }

    /**
     * Fügt eine Fehlermeldung hinzu, zu der kein Fehlerdialog angezeigt wird, sondern der Benutzer
     * nur informiert wird.
     */
    public void addNoDialogErrorMessage(String noDialogErrorMessage) {
        exceptionHandler.addNoDialogErrorMessage(noDialogErrorMessage);
    }

    /** Registriert den Handler im Swing-Thread ohne eine Aktion nach Anzeige des Fehlers. */
    public void registerToEdt() {
        try {
            SwingUtilities.invokeAndWait(() ->
                Thread.setDefaultUncaughtExceptionHandler(exceptionHandler));
        }
        catch (InvocationTargetException e) {
            e.printStackTrace();
        }
        catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    /** Getter für den ExceptionHandler, der hier registriert wird. */
    public GuiExceptionHandler getExceptionHandler() {
        return exceptionHandler;
    }

}
