package de.duehl.swing.ui.key;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Dieser Dialog fragt die Eingabe eines Tastaturbefehls ab.
 *
 * @version 1.01     2021-04-21
 * @author Christian Dühl
 */

public class KeyboardShortCutDialog extends ModalDialogBase {

    private static final boolean SHOW_OUTPUT_AREA = false;
    private static final boolean CLOSE_AFTER_PRESSED = true;

    private final JTextArea keyboardInputArea;
    private final JTextArea outputArea;

    /** Der zuletzt eingegebene Tastaturbefehl. */
    private KeyDefinition keyboardShortCut;

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Icon für das Programm.
     */
    public KeyboardShortCutDialog(Point parentLocation, Image programImage) {
        super(parentLocation, programImage, "Eingabe eines Keyboard-ShortCuts");
        setMinimumSize(500, 1);

        keyboardInputArea = new JTextArea();
        outputArea = new JTextArea();

        keyboardShortCut = KeyDefinition.NO_KEY_DEFINIED;

        initFields();

        fillDialog();

        keyboardInputArea.requestFocusInWindow();
    }

    /*
     * Achtung Umlaute!
     *
     * Da du als KeyCode für ä, ö und ü nur 0 bekommst, musst du über den KeyChar herausfinden,
     * welche Taste gedrückt wurde.
     *
     * zB so
     *
     * if (evt.getKeyChar() == 'ö') {
     *     doSomething();
     * }
     * else {
     *     doSomethingElse();
     * }
     *
     * wobei evt vom Typ KeyEvent ist.
     *
     * -> https://www.lima-city.de/thread/umlaute-in-java
     */

    private void initFields() {
        keyboardInputArea.setRows(5);
        keyboardInputArea.setColumns(40);
        keyboardInputArea.addKeyListener(new KeyListener() {
            @Override
            public void keyPressed(KeyEvent event) {
                int keyCode = event.getKeyCode();
                int modifiers = event.getModifiersEx();
                keyboardShortCut = new KeyDefinition(keyCode, modifiers);

                outputArea.append("PRESSED: " + keyboardShortCut.toReadableString() + "\n");
                outputArea.setCaretPosition(outputArea.getText().length());
            }

            @Override
            public void keyReleased(KeyEvent event) {
                int keyCode = event.getKeyCode();
                int modifiers = event.getModifiersEx();
                KeyDefinition releasedKeyboardShortCut = new KeyDefinition(keyCode, modifiers);

                outputArea.append("RELEASED: " + releasedKeyboardShortCut.toReadableString() + "\n");
                outputArea.setCaretPosition(outputArea.getText().length());

                if (CLOSE_AFTER_PRESSED) {
                    closeDialog();
                }
            }

            @Override
            public void keyTyped(KeyEvent event) {
                int keyCode = event.getKeyCode();
                int modifiers = event.getModifiersEx();
                KeyDefinition typedKeyboardShortCut = new KeyDefinition(keyCode, modifiers);

                outputArea.append("TYPED: " + typedKeyboardShortCut.toReadableString() + "\n");
                outputArea.setCaretPosition(outputArea.getText().length());
            }
        });

        outputArea.setRows(15);
        outputArea.setColumns(40);
        outputArea.setEditable( false );
        outputArea.setBackground( Color.LIGHT_GRAY );
    }

    @SuppressWarnings("unused")
    private String getLocation(KeyEvent e) {
        String loc = "";

        switch (e.getKeyLocation()) {
            case KeyEvent.KEY_LOCATION_STANDARD:
                loc = "STANDARD";
                break;
            case KeyEvent.KEY_LOCATION_LEFT:
                loc = "LEFT";
                break;
            case KeyEvent.KEY_LOCATION_RIGHT:
                loc = "RIGHT";
                break;
            case KeyEvent.KEY_LOCATION_NUMPAD:
                loc = "NUMPAD";
                break;
        }

        return loc;
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(new JScrollPane(keyboardInputArea), BorderLayout.NORTH);
        if (SHOW_OUTPUT_AREA) {
            add(new JScrollPane(outputArea), BorderLayout.CENTER);
        }
    }

    /** Getter für den zuletzt eingegebenen Tastaturbefehl. */
    public KeyDefinition getSelectedKeyDefinition() {
        return keyboardShortCut;
    }

}
