package de.duehl.swing.ui.tables;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;

import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;

import de.duehl.swing.ui.colors.ColorTool;

/**
 * Diese Klasse ist für das Rendern (die optische Darstellung) der Tabellenzellen zuständig. Sie
 * wurde von DefaultTableCellRenderer abgeleitet, um nicht für jede Zelle eine neue Klasse zu
 * erzeugen.
 *
 * @version 1.01     2020-01-17
 * @author Christian Dühl
 */

public class DifferentBackgroundsTableRenderer extends DefaultTableCellRenderer {

    private static final long serialVersionUID = 1L;

    public static final float FONT_SIZE = 15f;

    /** Normale Vordergrundfarbe. */
    private final Color lighterForeground;

    /** Normale Hintergrundfarbe. */
    private final Color lighterBackground;

    /** Dunklere Vordergrundfarbe. */
    private final Color darkerForeground;

    /** Dunklere Hintergrundfarbe. */
    private final Color darkerBackground;

    /**
     * Gibt an, ob bei selektierten Zeilen die Farben von Vorder- und Hintergrund vertauscht werden
     * sollen.
     */
    private boolean useSwitchedColorsForSelection;

    /**
     * Der Konstruktor, hier werden unsere Maps erzeugt.
     *
     * @param foregroundColor
     *            Vordergrundfarbe
     * @param backgroundColor
     *            Hintergrundfarbe
     */
    public DifferentBackgroundsTableRenderer(Color foregroundColor, Color backgroundColor) {
        super();

        this.lighterForeground = ColorTool.calculateLighterColor(foregroundColor);
        this.lighterBackground = ColorTool.calculateLighterColor(backgroundColor);

        this.darkerForeground = ColorTool.calculateDarkerColor(foregroundColor);
        this.darkerBackground = ColorTool.calculateDarkerColor(backgroundColor);

        useSwitchedColorsForSelection = true;
    }

    /**
     * Legt fest, dass für die Selection die gleichen Farben genutzt werden wir für die normale
     * Darstellung.
     */
    public void doNotUseSwitchedColorsForSelection() {
        useSwitchedColorsForSelection = false;
    }

    /**
     * Liefert eine Komponente für die anzuzeigende Zelle zurück. Die Hauptarbeit lassen wir
     * super.getTableCellRendererComponent(...) machen und hängen nur ein wenig Code dahinter.
     *
     * @param table
     *            Die Tabelle
     * @param value
     *            Das anzuzeigende Objekt (im Normalfall ein String)
     * @param isSelected
     *            Zeigt an, ob das Objekt selektiert wurde.
     * @param hasFocus
     *            Zeigt an, ob der Focus auf dieser Zelle liegt.
     * @param row
     *            Zeile in der die Zelle liegt.
     * @param column
     *            Spalte in der die Zelle liegt.
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
            boolean hasFocus, int row, int column) {
        JLabel createdLabel = (JLabel) super.getTableCellRendererComponent(table, value, isSelected,
                hasFocus, row, column);
        //createdLabel.setFont(new Font("Arial", Font.BOLD, 30));
        setVerticalAlignment(JLabel.CENTER);
        setHorizontalAlignment(JLabel.LEFT);


        Font oldFont = createdLabel.getFont();
        createdLabel.setFont(oldFont.deriveFont(FONT_SIZE));
        //c.setFont(new Font("Arial", Font.BOLD, 30));
        setVerticalAlignment(JLabel.CENTER);
        setHorizontalAlignment(JLabel.LEFT);

        //table.setRowHeight((int) (tableFontSize * 1.6));
        //table.setRowHeight(20);
        // Das geht nicht, das führt zu rekursiven Aufrufen dieser Methode,
        // wodurch die CPU-Last auf 100% geht.
        // Besser beim erstellen der Tabelle die Zeilenhöhe ändern:
        //     table.setRowHeight((int) (1.6 * tableFontSize));

        boolean odd = row % 2 == 0;
        Color foreground = odd ? lighterForeground : darkerForeground;
        Color background = odd ? lighterBackground : darkerBackground;
        if (isSelected && useSwitchedColorsForSelection) {
            setForeground(background);
            setBackground(foreground);
        }
        else {
            setForeground(foreground);
            setBackground(background);
        }

        return createdLabel;
    }

}
