package de.duehl.swing.text.html.characteristic;

import org.junit.Test;

import de.duehl.basics.text.handling.BooleanHandling;

import static de.duehl.swing.ui.colors.ColorTool.getHexColorByName;
import static de.duehl.swing.ui.colors.NamedColorListFabric.*;
import static org.junit.Assert.*;

import java.util.Properties;

public class OpticalTextPartCharacteristicTest {

    @Test
    public void createOpeningSpanDefault() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        String actual = characteristics.createOpeningSpan();
        String expected = "<span style=\"color:000000\">";
        assertEquals(expected, actual);
    }

    @Test
    public void createHtmlStyleDefault() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();

        String actual = characteristics.createHtmlStyle();
        String expected = "color:000000";
        assertEquals(expected, actual);
    }

    @Test
    public void createHtmlStyleBoldOrange() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        characteristics.setBold(true);
        characteristics.setForegroundHexColor(getHexColorByName(ORANGE));

        String actual = characteristics.createHtmlStyle();
        String expected = "color:ff6600;font-weight:bold";
        assertEquals(expected, actual);
    }

    @Test
    public void createHtmlStyleItalicSeagreen() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        characteristics.setItalic(true);
        characteristics.setForegroundHexColor(getHexColorByName(SEAGREEN));

        String actual = characteristics.createHtmlStyle();
        String expected = "color:2e8b57;font-style:italic";
        assertEquals(expected, actual);
    }

    @Test
    public void createHtmlStyleUnderlineBlackOnLightRed() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        characteristics.setForegroundHexColor(getHexColorByName(BLACK));
        characteristics.setBackgroundHexColor(getHexColorByName(LIGHTRED));
        characteristics.setUseBackgroundColor(true);
        characteristics.setUnderline(true);

        String actual = characteristics.createHtmlStyle();
        String expected = "color:000000;background-color:ffc8c8;text-decoration:underline";
        assertEquals(expected, actual);
    }

    @Test
    public void createHtmlStyleWithAll() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        characteristics.setForegroundHexColor(getHexColorByName(MEDIUMVIOLETRED));
        characteristics.setBackgroundHexColor(getHexColorByName(LIGHTCORAL));
        characteristics.setUseBackgroundColor(true);
        characteristics.setBold(true);
        characteristics.setItalic(true);
        characteristics.setUnderline(true);

        String actual = characteristics.createHtmlStyle();
        String expected = "color:c71585;"
                + "background-color:f08080;"
                + "font-weight:bold;"
                + "font-style:italic;"
                + "text-decoration:underline";
        assertEquals(expected, actual);
    }

    @Test
    public void createFullPropertyName() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        String actual = characteristics.createFullPropertyName("options", "foo", "bar");
        String expected = "options.foo.bar";
        assertEquals(expected, actual);
    }

    @Test
    public void loadCharacteristicsFromProperties() {
        Properties properties = new Properties();
        properties.setProperty("options.test.foregroundHexColor", "c71585");
        properties.setProperty("options.test.backgroundHexColor", "f08080");
        properties.setProperty("options.test.useBackgroundHexColor", "ja");
        properties.setProperty("options.test.bold", "ja");
        properties.setProperty("options.test.italic", "ja");
        properties.setProperty("options.test.underline", "ja");

        OpticalTextPartCharacteristic actualCharacteristics = new OpticalTextPartCharacteristic();
        actualCharacteristics.loadCharacteristicsFromProperties(properties, "options", "test");

        OpticalTextPartCharacteristic expectedCharacteristics = new OpticalTextPartCharacteristic();
        expectedCharacteristics.setForegroundHexColor(getHexColorByName(MEDIUMVIOLETRED));
        expectedCharacteristics.setBackgroundHexColor(getHexColorByName(LIGHTCORAL));
        expectedCharacteristics.setUseBackgroundColor(true);
        expectedCharacteristics.setBold(true);
        expectedCharacteristics.setItalic(true);
        expectedCharacteristics.setUnderline(true);

        assertEquals(expectedCharacteristics, actualCharacteristics);
    }

    @Test
    public void loadDefaulCharacteristicsFromProperties() {
        Properties properties = new Properties();
        properties.setProperty("options.test.foregroundHexColor", "000000");
        properties.setProperty("options.test.backgroundHexColor", "ffffff");
        properties.setProperty("options.test.useBackgroundHexColor", "nein");
        properties.setProperty("options.test.bold", "nein");
        properties.setProperty("options.test.italic", "nein");
        properties.setProperty("options.test.underline", "nein");

        OpticalTextPartCharacteristic actualCharacteristics = new OpticalTextPartCharacteristic();
        actualCharacteristics.loadCharacteristicsFromProperties(properties, "options", "test");

        OpticalTextPartCharacteristic expectedCharacteristics = new OpticalTextPartCharacteristic();

        assertEquals(expectedCharacteristics, actualCharacteristics);
    }

    @Test
    public void saveCharacteristicsToProperties() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        characteristics.setForegroundHexColor(getHexColorByName(MEDIUMVIOLETRED));
        characteristics.setBackgroundHexColor(getHexColorByName(LIGHTCORAL));
        characteristics.setUseBackgroundColor(true);
        characteristics.setBold(true);
        characteristics.setItalic(true);
        characteristics.setUnderline(true);

        Properties properties = new Properties();
        characteristics.saveCharacteristicsToProperties(properties, "options", "test");

        assertTrue(properties.containsKey("options.test.foregroundHexColor"));
        assertEquals("c71585", properties.getProperty("options.test.foregroundHexColor"));

        assertTrue(properties.containsKey("options.test.backgroundHexColor"));
        assertEquals("f08080", properties.getProperty("options.test.backgroundHexColor"));

        assertTrue(properties.containsKey("options.test.useBackgroundHexColor"));
        assertTrue(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.useBackgroundHexColor")));

        assertTrue(properties.containsKey("options.test.bold"));
        assertTrue(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.bold")));

        assertTrue(properties.containsKey("options.test.italic"));
        assertTrue(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.italic")));

        assertTrue(properties.containsKey("options.test.underline"));
        assertTrue(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.underline")));
    }

    @Test
    public void saveDefaultCharacteristicsToProperties() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();

        Properties properties = new Properties();
        characteristics.saveCharacteristicsToProperties(properties, "options", "test");

        assertTrue(properties.containsKey("options.test.foregroundHexColor"));
        assertEquals("000000", properties.getProperty("options.test.foregroundHexColor"));

        assertTrue(properties.containsKey("options.test.backgroundHexColor"));
        assertEquals("ffffff", properties.getProperty("options.test.backgroundHexColor"));

        assertTrue(properties.containsKey("options.test.useBackgroundHexColor"));
        assertFalse(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.useBackgroundHexColor")));

        assertTrue(properties.containsKey("options.test.bold"));
        assertFalse(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.bold")));

        assertTrue(properties.containsKey("options.test.italic"));
        assertFalse(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.italic")));

        assertTrue(properties.containsKey("options.test.underline"));
        assertFalse(BooleanHandling.stringToBoolean(
                properties.getProperty("options.test.underline")));
    }

    @Test
    public void copy() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        characteristics.setForegroundHexColor(getHexColorByName(MEDIUMVIOLETRED));
        characteristics.setBackgroundHexColor(getHexColorByName(LIGHTCORAL));
        characteristics.setUseBackgroundColor(true);
        characteristics.setBold(true);
        characteristics.setItalic(true);
        characteristics.setUnderline(true);

        assertEquals(getHexColorByName(MEDIUMVIOLETRED), characteristics.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTCORAL), characteristics.getBackgroundHexColor());
        assertTrue(characteristics.isUseBackgroundColor());
        assertTrue(characteristics.isBold());
        assertTrue(characteristics.isItalic());
        assertTrue(characteristics.isUnderline());

        OpticalTextPartCharacteristic copy = characteristics.copy();
        copy.setForegroundHexColor(getHexColorByName(SEAGREEN));
        copy.setBackgroundHexColor(getHexColorByName(LIGHTGOLDENRODYELLOW));
        copy.setUseBackgroundColor(false);
        copy.setBold(false);
        copy.setItalic(false);
        copy.setUnderline(false);

        assertEquals(getHexColorByName(MEDIUMVIOLETRED), characteristics.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTCORAL), characteristics.getBackgroundHexColor());
        assertTrue(characteristics.isUseBackgroundColor());
        assertTrue(characteristics.isBold());
        assertTrue(characteristics.isItalic());
        assertTrue(characteristics.isUnderline());

        assertEquals(getHexColorByName(SEAGREEN), copy.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTGOLDENRODYELLOW), copy.getBackgroundHexColor());
        assertFalse(copy.isUseBackgroundColor());
        assertFalse(copy.isBold());
        assertFalse(copy.isItalic());
        assertFalse(copy.isUnderline());
    }

    @Test
    public void copyFrom() {
        OpticalTextPartCharacteristic characteristics = new OpticalTextPartCharacteristic();
        characteristics.setForegroundHexColor(getHexColorByName(MEDIUMVIOLETRED));
        characteristics.setBackgroundHexColor(getHexColorByName(LIGHTCORAL));
        characteristics.setUseBackgroundColor(true);
        characteristics.setBold(true);
        characteristics.setItalic(true);
        characteristics.setUnderline(true);

        assertEquals(getHexColorByName(MEDIUMVIOLETRED), characteristics.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTCORAL), characteristics.getBackgroundHexColor());
        assertTrue(characteristics.isUseBackgroundColor());
        assertTrue(characteristics.isBold());
        assertTrue(characteristics.isItalic());
        assertTrue(characteristics.isUnderline());

        OpticalTextPartCharacteristic source = new OpticalTextPartCharacteristic();
        source.setForegroundHexColor(getHexColorByName(SEAGREEN));
        source.setBackgroundHexColor(getHexColorByName(LIGHTGOLDENRODYELLOW));
        source.setUseBackgroundColor(false);
        source.setBold(false);
        source.setItalic(false);
        source.setUnderline(false);

        assertEquals(getHexColorByName(SEAGREEN), source.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTGOLDENRODYELLOW), source.getBackgroundHexColor());
        assertFalse(source.isUseBackgroundColor());
        assertFalse(source.isBold());
        assertFalse(source.isItalic());
        assertFalse(source.isUnderline());

        characteristics.copyFrom(source);

        assertEquals(getHexColorByName(SEAGREEN), characteristics.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTGOLDENRODYELLOW), characteristics.getBackgroundHexColor());
        assertFalse(characteristics.isUseBackgroundColor());
        assertFalse(characteristics.isBold());
        assertFalse(characteristics.isItalic());
        assertFalse(characteristics.isUnderline());

        assertEquals(getHexColorByName(SEAGREEN), source.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTGOLDENRODYELLOW), source.getBackgroundHexColor());
        assertFalse(source.isUseBackgroundColor());
        assertFalse(source.isBold());
        assertFalse(source.isItalic());
        assertFalse(source.isUnderline());
    }

    @Test
    public void copyConstructor() {
        OpticalTextPartCharacteristic original = new OpticalTextPartCharacteristic();
        original.setForegroundHexColor(getHexColorByName(MEDIUMVIOLETRED));
        original.setBackgroundHexColor(getHexColorByName(LIGHTCORAL));
        original.setUseBackgroundColor(true);
        original.setBold(true);
        original.setItalic(true);
        original.setUnderline(true);

        assertEquals(getHexColorByName(MEDIUMVIOLETRED), original.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTCORAL), original.getBackgroundHexColor());
        assertTrue(original.isUseBackgroundColor());
        assertTrue(original.isBold());
        assertTrue(original.isItalic());
        assertTrue(original.isUnderline());

        OpticalTextPartCharacteristic copied = new OpticalTextPartCharacteristic(original);

        assertEquals(getHexColorByName(MEDIUMVIOLETRED), copied.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTCORAL), copied.getBackgroundHexColor());
        assertTrue(copied.isUseBackgroundColor());
        assertTrue(copied.isBold());
        assertTrue(copied.isItalic());
        assertTrue(copied.isUnderline());

        copied.setForegroundHexColor(getHexColorByName(SEAGREEN));
        copied.setBackgroundHexColor(getHexColorByName(LIGHTGOLDENRODYELLOW));
        copied.setUseBackgroundColor(false);
        copied.setBold(false);
        copied.setItalic(false);
        copied.setUnderline(false);

        assertEquals(getHexColorByName(MEDIUMVIOLETRED), original.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTCORAL), original.getBackgroundHexColor());
        assertTrue(original.isUseBackgroundColor());
        assertTrue(original.isBold());
        assertTrue(original.isItalic());
        assertTrue(original.isUnderline());

        assertEquals(getHexColorByName(SEAGREEN), copied.getForegroundHexColor());
        assertEquals(getHexColorByName(LIGHTGOLDENRODYELLOW), copied.getBackgroundHexColor());
        assertFalse(copied.isUseBackgroundColor());
        assertFalse(copied.isBold());
        assertFalse(copied.isItalic());
        assertFalse(copied.isUnderline());
    }

}
