package de.duehl.vocabulary.japanese.common.persistence;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.util.List;

import de.duehl.basics.system.SystemTools;
import de.duehl.basics.version.Version;
import de.duehl.swing.persistance.SwingSessionManagerHelper;
import de.duehl.vocabulary.japanese.common.data.TranslationDirection;
import de.duehl.vocabulary.japanese.common.data.VocablesSubsetSelectionType;
import de.duehl.vocabulary.japanese.common.data.VocablesShuffleType;
import de.duehl.vocabulary.japanese.common.data.VocabularySortOrder;
import de.duehl.vocabulary.japanese.common.persistence.data.EmptyTranslationsAcceptance;
import de.duehl.vocabulary.japanese.common.persistence.data.HistoricalOwnListPersistanceDataList;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.color.data.ColorModificationType;
import de.duehl.vocabulary.japanese.common.color.data.DefaultColors;

/**
 * Diese Klasse hält die persistenten Optionen und bietet Funktionen um sie abzufragen und zu
 * setzen.
 *
 * @version 1.01     2025-09-14
 * @author Christian Dühl
 */

public class Options implements KanjiAndKanaColorOptions, RomajiAndPronunciationColorOptions,
        TranslationCommentAndVocabularyDescriptionColorOptions {

    /** Schriftgröße für die StringSelections in der Oberfläche. */
    private int stringSelectionFontSize;

    /** Schriftgröße für die Titel der StringSelections in der Oberfläche. */
    private int stringSelectionLabelFontSize;

    /** Die Programmversion beim letzten Aufruf des Programms. */
    private Version lastUsedVersion;

    /** Der Pfad zu den Vokabularien und den Sound-Files. */
    private String vocabulariesPath;

    /** Gibt an, ob nach der Eingabe der Übersetzung gleich zur nächsten Vokabel gesprungen wird. */
    private boolean switchToNextVocableAfterEntringTranslation;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Romaji mit angezeigt werden soll.
     */
    private boolean showRomajiWhenTestingVocable;

    /** Gibt an, ob beim Abfragen der Vokabeln die Aussprache mit angezeigt werden soll. */
    private boolean showPronunciationWhenTestingVocable;

    /** Gibt an, ob beim Abfragen der Vokabeln der Kommentar mit angezeigt werden soll. */
    private boolean showCommentWhenTestingVocable;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Beschreibung des Vokabulars mit angezeigt werden
     * soll.
     */
    private boolean showVocabularyDescriptionWhenTestingVocable;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    private boolean colorVocableDependingOnLastSuccessWhenTestingVocable;

    /**
     * Gibt an, ob beim Anzeigen der Vokabeln die Darstellung abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    private boolean colorVocableDependingOnLastSuccessWhenViewingVocable;

    /** Die Art die Vokabularien in der Oberfläche zu sortieren. */
    private VocabularySortOrder vocabularySortOrder;

    /**
     * Gibt an, ob die Darstellung der angezeigten Vokabularien abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    private boolean colorVocabularyDependingOnLastSuccess;

    /** Die Art Vokabellisten zu sortieren. */
    private VocablesShuffleType vocablesShuffleType;

    /** Die Art eine Teilmenge von Vokabellisten zu bilden. */
    private VocablesSubsetSelectionType vocablesSubsetSelectionType;

    /** Gibt an, ob die Sortierung von Vokabellisten umgedreht werden soll. */
    private boolean reverseShuffledVocables;

    /**
     * Gibt an, ob die Sortierung von Vokabellisten auch zum Anzeigen der Listen oder Blätter
     * verwendet werden soll.
     */
    private boolean useVocablesShuffleTypeForShowingListsAndSheetsToo;

    /**
     * Gibt an, ob der Anfang der Beschreibung der Vokabularien ("B1_K04_1_" oder "X_") ausgeblendet
     * werden soll.
     */
    private boolean hideStartOfVocabularyDescription;

    /**
     * Gibt an, ob zum Betrachten der Vokabeln die Reihenfolge zum Abfragen verwendet werden soll.
     */
    private boolean useTestOrderForViewingVocables;

    /**
     * Der Index des zuletzt betrachteten Reiters (Vokabularien / eigene Listen) in der Oberfläche.
     */
    private int lastShownMainTabIndex;

    /**
     * Der Index des zuletzt betrachteten Kategorie-Reiters (Vokabularien / eigene Listen) in der
     * Oberfläche.
     */
    private int lastShownVocabularyCategoryTabIndex;

    /**
     * Der Index des zuletzt betrachteten Kategorie-Reiters (Vokabularien / eigene Listen) in der
     * Oberfläche.
     */
    private int lastShownOwnListCategoryTabIndex;

    /**
     * Der Index des zuletzt betrachteten Unterkategorie-Reiters (Vokabularien / eigene Listen) in
     * der Oberfläche.
     */
    private int lastShownVocabularySubCategoryTabIndex;

    /**
     * Der Index des zuletzt betrachteten Unterkategorie-Reiters (Vokabularien / eigene Listen) in
     * der Oberfläche.
     */
    private int lastShownOwnListSubCategoryTabIndex;

    /**
     * Gibt an, ob (ggf. neben der Farbe) der Erfolg bei den letzten zehn Abfragen der Vokabeln
     * eines Vokabulars als Prozentzahl mit angezeigt werden soll.
     */
    private boolean showSuccessPercentInVocabularyBar;

    /** Die Anzahl der Spalten mit den Vokabularien. */
    private int numberOfVocabularyBarColumns;

    /** Das zuletzt zum Im- oder Exportieren von Vokabellisten verwendete Verzeichnis. */
    private String lastUsedOwnListImportExportDirectory;

    /**
     * Die Anzahl an Datensätzen pro Seite im Editor, in welchem eine eigene Listen von Vokabeln
     * bearbeitet wird.
     */
    private int numberOfDatasetsPerPageOfOwnListEditor;

    /**
     * Die Anzahl an Spalten mit den auf einer Seite angezeigten Daten im Editor, in welchem eine
     * eigene Listen von Vokabeln bearbeitet wird.
     */
    private int numberOfColumnsOfOwnListEditor;

    /** Die Position des Editors, in welchem eine eigene Listen von Vokabeln bearbeitet wird. */
    private Point locationOfOwnListEditor;

    /** Die Größe des Editors, in welchem eine eigene Listen von Vokabeln bearbeitet wird. */
    private Dimension sizeOfOwnListEditor;

    /**
     * Gibt an, ob Vokabeln aus der Liste rechts im Suchbereich versteckt werden sollen. Falls
     * nicht, werden sie nur anders dargestellt.
     */
    private boolean hideVocablesFromListInSearchAreaInOwnListEditor;

    /** Gibt an, ob das Programm bei fehlenden MP3-Dateien gestartet werden soll. */
    private boolean allowMissingMp3;

    /** Gibt an, ob das Programm beim Start auf fehlende MP3-Dateien hinweisen soll. */
    private boolean reportMissingMp3;

    /** Die Anzahl an Spalten in der Ansicht als Vokabelblatt. */
    private int numberOfColumnsOfBlattDarstellung;

    /**
     * Gibt an, ob beim Start des Programms über doppelte MP3-Dateien mit dem gleichen Namen
     * informiert werden soll. Diese gibt es teilweise aus berechtigten Gründen, aber es kann auch
     * auf Fehler hinweisen.
     */
    private boolean informAboutDoubleMp3AtStartup;

    /**
     * Gibt an, ob bei der Übersetzung von Deutsch nach Japanisch nur der erste deutsche Begriff,
     * welcher in der Vokabel unter translations hinterlegt ist, abgefragt wird. Anderenfalls wird
     * zufällig irgendeiner der deutschen Begriffe abgefragt.
     */
    private boolean useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese;

    /** Die Richtung, in die übersetzt wird (Japanisch - Deutsch oder Deutsch - Japanisch). */
    private TranslationDirection translationDirection;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Kanji mit angezeigt werden soll.
     */
    private boolean showKanjiWhenTestingVocable;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Kanji weggelassen werden soll,
     * falls die Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    private boolean hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Romaji weggelassen werden soll,
     * falls die Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    private boolean hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Aussprache weggelassen werden soll, falls die
     * Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    private boolean hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana;

    /** Gibt an, ob der Button zum Anzeigen der Details in der Einzeldarstellung angezeigt wird. */
    private boolean showDetailsButtonInEinzeldarstellung;

    /**
     * Gibt an, ob der Button zum Anzeigen der Details in der Darstellung als Vokabelblatt
     * angezeigt wird.
     */
    private boolean showDetailsButtonInVokabelblattDarstellung;

    /**
     * Gibt an, ob bei Vokabularien und Listen aus der VHS weniger vom Anfang der Beschreibung der
     * Vokabularien (entfernt "VHS_A1.1_" statt "VHS_A1.1_01_01_") ausgeblendet werden soll.
     */
    private boolean hideLessStartOfVocabularyDescriptionForVhs;

    /** Gibt an, ob beim Start des Programms über gleiche Kana und Kanji informiert werden soll. */
    private boolean informAboutEqualKanaAndKanjiAtStartup;

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Kana mit angezeigt werden soll.
     */
    private boolean showKanaWhenTestingVocable;

    /**
     * Gibt an, ob beim Start des Programms auf eine aktuellere Version der eigenen Listen geprüft
     * werden soll.
     */
    private boolean checkNewOwnListsVersionAtStart;

    /**
     * Gibt an, ob die Kana, wenn auch Kanji dargestellt werden, beim Abfragen genauso groß wie die
     * Kanji dargestellt werden sollen. Anderenfalls werden sie etwas kleiner dargestellt.
     */
    private boolean showKanaInKanjiSizeWhenTestingVocable;

    /** Die Art, ob auf das Abschicken einer leeren Übersetzung reagiert werden soll. */
    private EmptyTranslationsAcceptance emptyTranslationsAcceptance;

    /**
     * Gibt an, wieviele Tage 'eine Weile' ist, um Vokabeln anzuzeigen oder abzufragen, die 'lange
     * nicht mehr abgefragt' wurden.
     */
    private int numberOfDaysForAWhile;

    /**
     * Gibt an wieviele Tage seit des ersten Auftretens eine Vokabel noch als 'neu' gilt, um die
     * 'neuesten' Vokabeln abzufragen oder anzuzeigen.
     */
    private int numberOfDaysAVocableIsNew;

    /** Die Liste der elf Farben von schlecht bis gut (0/10, 1/10 bis 10/10 richtige Antworten). */
    private List<Color> colors;

    /** Der Wert um den die Farben für eine Vordergrundfarbe abgedunkelt werden. */
    private int deltaForForegroundColor;

    /** Der Wert um den die Farben für eine Hintergrundfarbe aufgehellt werden. */
    private int deltaForBackgroundColor;

    /**
     * Die Art wie Farben für die Erzeugung von Vorder- bzw. Hintergrundfarben modifiziert werden.
     */
    private ColorModificationType colorModificationType;

    /**
     * Gibt an, ob im Dialog zur Bearbeitung einer eigenen Liste ein variabler Trenner zwischen dem
     * linken Teil mit der Auswahl der Vokabeln und dem rechten Teil mit der Liste verwendet wird.
     */
    private boolean useSplitPaneInOwnListEditor;

    /** Die Farbe für normalen Text bei der Darstellung einer Vokabel. */
    private Color vocableColorStandard;

    /** Die Farbe für Kanji bei der Darstellung einer Vokabel. */
    private Color vocableColorKanji;

    /** Die Farbe für Kana bei der Darstellung einer Vokabel. */
    private Color vocableColorKana;

    /** Die Farbe für die Romaji bei der Darstellung einer Vokabel. */
    private Color vocableColorRomaji;

    /** Die Farbe für die Aussprache bei der Darstellung einer Vokabel. */
    private Color vocableColorPronunciation;

    /** Die Farbe für die Übersetzung bei der Darstellung einer Vokabel. */
    private Color vocableColorTranslation;

    /** Die Farbe für den Kommentar bei der Darstellung einer Vokabel. */
    private Color vocableColorComment;

    /** Die Farbe für einen Suchbegriff bei der Darstellung einer Vokabel. */
    private Color vocableColorSearchWords;

    /** Die Farbe für eine Wortart bei der Darstellung einer Vokabel. */
    private Color vocableColorPartOfSpeach;

    /**
     * Gibt an, ob beim Startup die bis zu diesem Schritt bereits benötigte Laufzeit vor den
     * Meldungen ausgegeben werden soll.
     */
    private boolean showTimestampsInStartup;

    /**
     * Gibt an, ob das Verzeichnis für die Übersetzung Deutsch nach Japanisch beim Start aufgebaut
     * werden soll. Dies dauert einige Zeit, die der Start länger dauert. Im anderen Fall wird
     * dieses Verzeichnis erst aufgebaut, wenn es das erste Mal benötigt wird, also wenn die erste
     * Vokabel von Deutsch nach Japanisch übersetzt wird.
     *
     * Übersetzt man oft auch von Deutsch nach Japanisch, sollte man es beim Startup erzeugen,
     * anderenfalls ist es besser, es erst dann zu erzeugen, wenn man es braucht, weil man es
     * meistens nicht braucht und so der Vokabel-Trainer schneller startet.
     */
    private boolean createGermanJapaneseTranslationAtStartup;

    /** Das zuletzt zum Auspacken verwendete 7Zip-Executable. */
    private String lastUsed7ZipDirectory;

    /**
     * Der zuletzt ausgewählte Text der Combobox zur Auswahl der abzufragenden Menge an Kanji im
     * KanjiTestParameterChooser.
     */
    private String lastSelectedKanjiSetTextInCombobox;

    /**
     * Gibt an, ob beim Start geprüft werden soll, ob "e" oder "u" in der Aussprache der Vokabeln
     * vorkommt. Dies ist nur sinnvoll für jemanden, der die Vokabeln direkt bearbeitet. Sinn
     * dahinter ist, dass ich diese Vokabel in der Aussprache einheitlich als "ä" und "ü"
     * darstellen möchte.
     */
    private boolean informAboutEOrUInVocableFieldAusspracheAtStartup;

    /**
     * Die Anzahl der in der Historie im Listen-Commander links und rechts sowie beim Hinzufügen
     * von Vokabeln zu eigenen Listen über den Detail-Dialog angezeigten besuchten Listen.
     */
    private int numberOfRememberedOwnLists;

    /** Die zuletzt auf der linken Seite des List-Commanders genutzten eigenen Listen. */
    private HistoricalOwnListPersistanceDataList historicalLeftOwnListsInListCommander;

    /** Die zuletzt auf der rechten Seite des List-Commanders genutzten eigenen Listen. */
    private HistoricalOwnListPersistanceDataList historicalRightOwnListsInListCommander;

    /** Die zuletzt außerhalb des List-Commanders genutzten eigenen Listen. */
    private HistoricalOwnListPersistanceDataList historicalOwnLists;

    /**
     * Gibt an, ob in der Einzeldarstellung von Vokabeln und im Dialog zur Bewertung einer
     * Übersetzung alle Übersetzungen angezeigt werden.
     *
     * Dies kann bei vielen Übersetzungen zur Überlänge führen, so dass die Dialoge nach unten aus
     * dem Bildschirm ragen.
     */
    private boolean showAllTranslations;

    /**
     * Gibt an, wieviele Übersetzungen in der Einzeldarstellung von Vokabeln und im Dialog zur
     * Bewertung einer Übersetzung angezeigt werden, falls denn der Benutzer nicht alle
     * Übersetzungen anzeigen möchte (siehe showAllTranslations).
     *
     * Die Anzeige aller oder zu vieler Übersetzungen kann bei vielen Übersetzungen zur Überlänge
     * führen, so dass die Dialoge nach unten aus dem Bildschirm ragen. Mit dieser Einstellung kann
     * man abhängig von der Monitorauflösung eine passende Wahl treffen.
     */
    private int numberOfShownTranslations;

    /**
     * Gibt an, ob die Anzahl der Listen, in der die Vokabel ist, zusammen mit einem Hinweis auf
     * Lernlisten, in der Listendarstellung eines Vokabulars oder einer eigenen Liste angezeigt
     * werden soll.
     */
    private boolean showNumberOfListsInLister;

    /**
     * Gibt an, ob die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum
     * Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen der bekannten
     * Verb- bzw. Adjektivformen, auch im Dialog mit der Einzeldarstellung angezeigt werden sollen.
     */
    private boolean showDetailBottomButonsInEinzeldarstellungToo;

    /**
     * Gibt an, ob die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum
     * Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen der bekannten
     * Verb- bzw. Adjektivformen, auch im Dialog mit der Bewertung einer einzelnen Abfrage einer
     * Vokabel angezeigt werden sollen.
     */
    private boolean showDetailBottomButonsInTranslationEvaluationToo;

    /** Gibt an, ob beim Abfragen der Vokabeln diese automatisch vorgelesen werden sollen. */
    private boolean playMP3WhenTestingVocables;

    /**
     * Gibt an, ob beim Abfragen der Kanji werden die Felder der Benutzereingaben im Dialog,
     * welcher Erfolg oder Misserfolg der Angaben zu einem einzelnen Kanji anzeigt, rot oder grün
     * eingefärbt.
     */
    private boolean colorUserInputAtKanjiTestEvaluationDialog;

    /**
     * Konstruktor: Initialisiert mit Defaultwerten.
     *
     * @param vocabularyTrainerVersion
     *            Die Version des Vokabel-Trainers (erhältlich mit
     *            new VocabularyTrainerVersion().getVersion()).
     */
    public Options(Version vocabularyTrainerVersion) {
        stringSelectionFontSize = -1;
        stringSelectionLabelFontSize = -1;
        lastUsedVersion = vocabularyTrainerVersion;
        vocabulariesPath = "";
        switchToNextVocableAfterEntringTranslation = true;
        showRomajiWhenTestingVocable = true;
        showPronunciationWhenTestingVocable = true;
        showCommentWhenTestingVocable = false;
        showVocabularyDescriptionWhenTestingVocable = false;
        colorVocableDependingOnLastSuccessWhenTestingVocable = true;
        colorVocableDependingOnLastSuccessWhenViewingVocable = true;
        vocabularySortOrder = VocabularySortOrder.ALPHABETICAL;
        colorVocabularyDependingOnLastSuccess = true;
        vocablesShuffleType = VocablesShuffleType.RANDOM;
        vocablesSubsetSelectionType = VocablesSubsetSelectionType.RANDOM_AND_LAST_ANSWER_SUCCESS;
        reverseShuffledVocables = false;
        useVocablesShuffleTypeForShowingListsAndSheetsToo = false;
        hideStartOfVocabularyDescription = true;
        useTestOrderForViewingVocables = false;
        lastShownMainTabIndex = -1;
        lastShownVocabularyCategoryTabIndex = -1;
        lastShownOwnListCategoryTabIndex = -1;
        lastShownVocabularySubCategoryTabIndex = -1;
        lastShownOwnListSubCategoryTabIndex = -1;
        showSuccessPercentInVocabularyBar = true;
        numberOfVocabularyBarColumns = 3;
        lastUsedOwnListImportExportDirectory = SystemTools.getHomeDirectory();
        numberOfDatasetsPerPageOfOwnListEditor = 20;
        numberOfColumnsOfOwnListEditor = 2;
        locationOfOwnListEditor = SwingSessionManagerHelper.NOT_INITIALISED_POINT;
        sizeOfOwnListEditor = SwingSessionManagerHelper.NOT_INITIALISED_DIMENSION;
        hideVocablesFromListInSearchAreaInOwnListEditor = false;
        allowMissingMp3 = true;
        reportMissingMp3 = false;
        numberOfColumnsOfBlattDarstellung = 3;
        informAboutDoubleMp3AtStartup = false;
        useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese = true;
        translationDirection = TranslationDirection.JAPANESE_TO_GERMAN;
        showKanjiWhenTestingVocable = true;
        hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana = false;
        hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana = false;
        hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana = false;
        showDetailsButtonInEinzeldarstellung = false;
        showDetailsButtonInVokabelblattDarstellung = false;
        hideLessStartOfVocabularyDescriptionForVhs = true;
        informAboutEqualKanaAndKanjiAtStartup = false;
        showKanaWhenTestingVocable = true;
        checkNewOwnListsVersionAtStart = true;
        showKanaInKanjiSizeWhenTestingVocable = false;
        emptyTranslationsAcceptance = EmptyTranslationsAcceptance.NOT_IMMEDIATELY;
        numberOfDaysForAWhile = 30;
        numberOfDaysAVocableIsNew = 7;
        colors = DefaultColors.INTENSIVE_COLORS;
        deltaForForegroundColor = VocableColors.DEFAULT_DELTA_FOR_FOREGROUND_COLOR;
        deltaForBackgroundColor = VocableColors.DEFAULT_DELTA_FOR_BACKGROUND_COLOR;
        colorModificationType = ColorModificationType.MULTIPLICATIVE;
        useSplitPaneInOwnListEditor = true;
        vocableColorStandard = Color.BLACK;
        vocableColorKanji = Color.RED;
        vocableColorKana = Color.BLUE;
        vocableColorRomaji = Color.BLACK;
        vocableColorPronunciation = Color.DARK_GRAY;
        vocableColorTranslation = Color.RED;
        vocableColorComment = Color.BLUE;
        vocableColorSearchWords = new Color(0, 160, 160);
        vocableColorPartOfSpeach = new Color(0, 180, 20);
        showTimestampsInStartup = false;
        createGermanJapaneseTranslationAtStartup = true;
        lastUsed7ZipDirectory = "";
        lastSelectedKanjiSetTextInCombobox = "";
        informAboutEOrUInVocableFieldAusspracheAtStartup = false;
        numberOfRememberedOwnLists = 10;
        historicalLeftOwnListsInListCommander = new HistoricalOwnListPersistanceDataList(
                numberOfRememberedOwnLists);
        historicalRightOwnListsInListCommander = new HistoricalOwnListPersistanceDataList(
                numberOfRememberedOwnLists);
        historicalOwnLists = new HistoricalOwnListPersistanceDataList(
                numberOfRememberedOwnLists);
        showAllTranslations = false;
        numberOfShownTranslations = 10;
        showNumberOfListsInLister = false;
        showDetailBottomButonsInEinzeldarstellungToo = true;
        showDetailBottomButonsInTranslationEvaluationToo = true;
        playMP3WhenTestingVocables = false;
        colorUserInputAtKanjiTestEvaluationDialog = true;
    }

    /** Getter für die Schriftgröße für die StringSelections in der Oberfläche. */
    public int getStringSelectionFontSize() {
        return stringSelectionFontSize;
    }

    /** Setter für die Schriftgröße für die StringSelections in der Oberfläche. */
    public void setStringSelectionFontSize(int stringSelectionFontSize) {
        this.stringSelectionFontSize = stringSelectionFontSize;
    }

    /** Getter für die Schriftgröße für die Titel der StringSelections in der Oberfläche. */
    public int getStringSelectionLabelFontSize() {
        return stringSelectionLabelFontSize;
    }

    /** Setter für die Schriftgröße für die Titel der StringSelections in der Oberfläche. */
    public void setStringSelectionLabelFontSize(int stringSelectionLabelFontSize) {
        this.stringSelectionLabelFontSize = stringSelectionLabelFontSize;
    }

    /** Getter für die Programmversion beim letzten Aufruf des Programms. */
    public Version getLastUsedVersion() {
        return lastUsedVersion;
    }

    /** Setter für die Programmversion beim letzten Aufruf des Programms. */
    public void setLastUsedVersion(Version lastUsedVersion) {
        this.lastUsedVersion = lastUsedVersion;
    }

    /** Getter für den Pfad zu den Vokabularien und den Sound-Files. */
    public String getVocabulariesPath() {
        return vocabulariesPath;
    }

    /** Setter für den Pfad zu den Vokabularien und den Sound-Files. */
    public void setVocabulariesPath(String vocabulariesPath) {
        this.vocabulariesPath = vocabulariesPath;
    }

    /** Gibt an, ob nach der Eingabe der Übersetzung gleich zur nächsten Vokabel gesprungen wird. */
    public boolean isSwitchToNextVocableAfterEntringTranslation() {
        return switchToNextVocableAfterEntringTranslation;
    }

    /**
     * Legt fest, ob nach der Eingabe der Übersetzung gleich zur nächsten Vokabel gesprungen wird.
     */
    public void setSwitchToNextVocableAfterEntringTranslation(
            boolean switchToNextVocableAfterEntringTranslation) {
        this.switchToNextVocableAfterEntringTranslation =
                switchToNextVocableAfterEntringTranslation;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Romaji mit angezeigt werden soll.
     */
    public boolean isShowRomajiWhenTestingVocable() {
        return showRomajiWhenTestingVocable;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Darstellung in Romaji mit angezeigt werden soll.
     */
    public void setShowRomajiWhenTestingVocable(boolean showRomajiWhenTestingVocable) {
        this.showRomajiWhenTestingVocable = showRomajiWhenTestingVocable;
    }

    /** Gibt an, ob beim Abfragen der Vokabeln die Aussprache mit angezeigt werden soll. */
    public boolean isShowPronunciationWhenTestingVocable() {
        return showPronunciationWhenTestingVocable;
    }

    /** Legt fest, ob beim Abfragen der Vokabeln die Aussprache mit angezeigt werden soll. */
    public void setShowPronunciationWhenTestingVocable(
            boolean showPronunciationWhenTestingVocable) {
        this.showPronunciationWhenTestingVocable = showPronunciationWhenTestingVocable;
    }

    /** Gibt an, ob beim Abfragen der Vokabeln der Kommentar mit angezeigt werden soll. */
    public boolean isShowCommentWhenTestingVocable() {
        return showCommentWhenTestingVocable;
    }

    /** Legt fest, ob beim Abfragen der Vokabeln der Kommentar mit angezeigt werden soll. */
    public void setShowCommentWhenTestingVocable(boolean showCommentWhenTestingVocable) {
        this.showCommentWhenTestingVocable = showCommentWhenTestingVocable;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Beschreibung des Vokabulars mit angezeigt werden
     * soll.
     */
    public boolean isShowVocabularyDescriptionWhenTestingVocable() {
        return showVocabularyDescriptionWhenTestingVocable;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Beschreibung des Vokabulars mit angezeigt
     * werden soll.
     */
    public void setShowVocabularyDescriptionWhenTestingVocable(
            boolean showVocabularyDescriptionWhenTestingVocable) {
        this.showVocabularyDescriptionWhenTestingVocable =
                showVocabularyDescriptionWhenTestingVocable;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    public boolean isColorVocableDependingOnLastSuccessWhenTestingVocable() {
        return colorVocableDependingOnLastSuccessWhenTestingVocable;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Darstellung abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    public void setColorVocableDependingOnLastSuccessWhenTestingVocable(
            boolean colorVocableDependingOnLastSuccessWhenTestingVocable) {
        this.colorVocableDependingOnLastSuccessWhenTestingVocable =
                colorVocableDependingOnLastSuccessWhenTestingVocable;
    }

    /**
     * Gibt an, ob beim Anzeigen der Vokabeln die Darstellung abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    public boolean isColorVocableDependingOnLastSuccessWhenViewingVocable() {
        return colorVocableDependingOnLastSuccessWhenViewingVocable;
    }

    /**
     * Legt fest, ob beim Anzeigen der Vokabeln die Darstellung abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    public void setColorVocableDependingOnLastSuccessWhenViewingVocable(
            boolean colorVocableDependingOnLastSuccessWhenViewingVocable) {
        this.colorVocableDependingOnLastSuccessWhenViewingVocable =
                colorVocableDependingOnLastSuccessWhenViewingVocable;
    }

    /** Getter für die Art die Vokabularien in der Oberfläche zu sortieren. */
    public VocabularySortOrder getVocabularySortOrder() {
        return vocabularySortOrder;
    }

    /** Setter für die Art die Vokabularien in der Oberfläche zu sortieren. */
    public void setVocabularySortOrder(VocabularySortOrder vocabularySortOrder) {
        this.vocabularySortOrder = vocabularySortOrder;
    }

    /**
     * Gibt an, ob die Darstellung der angezeigten Vokabularien abhängig vom Erfolg bei den letzten
     * zehn Abfragen eingefärbt werden soll.
     */
    public boolean isColorVocabularyDependingOnLastSuccess() {
        return colorVocabularyDependingOnLastSuccess;
    }

    /**
     * Legt fest, ob die Darstellung der angezeigten Vokabularien abhängig vom Erfolg bei den
     * letzten zehn Abfragen eingefärbt werden soll.
     */
    public void setColorVocabularyDependingOnLastSuccess(
            boolean colorVocabularyDependingOnLastSuccess) {
        this.colorVocabularyDependingOnLastSuccess = colorVocabularyDependingOnLastSuccess;
    }

    /** Getter für die Art Vokabellisten zu sortieren. */
    public VocablesShuffleType getVocablesShuffleType() {
        return vocablesShuffleType;
    }

    /** Setter für die Art Vokabellisten zu sortieren. */
    public void setVocablesShuffleType(VocablesShuffleType vocablesShuffleType) {
        this.vocablesShuffleType = vocablesShuffleType;
    }

    /** Getter für die Art eine Teilmenge von Vokabellisten zu bilden. */
    public VocablesSubsetSelectionType getVocablesSubsetSelectionType() {
        return vocablesSubsetSelectionType;
    }

    /** Setter für die Art eine Teilmenge von Vokabellisten zu bilden. */
    public void setVocablesSubsetSelectionType(
            VocablesSubsetSelectionType vocablesSubsetSelectionType) {
        this.vocablesSubsetSelectionType = vocablesSubsetSelectionType;
    }

    /** Gibt an, ob die Sortierung von Vokabellisten umgedreht werden soll. */
    public boolean isReverseShuffledVocables() {
        return reverseShuffledVocables;
    }

    /** Legt fest, ob die Sortierung von Vokabellisten umgedreht werden soll. */
    public void setReverseShuffledVocables(boolean reverseShuffledVocables) {
        this.reverseShuffledVocables = reverseShuffledVocables;
    }

    /**
     * Gibt an, ob die Sortierung von Vokabellisten auch zum Anzeigen der Listen oder Blätter
     * verwendet werden soll.
     */
    public boolean isUseVocablesShuffleTypeForShowingListsAndSheetsToo() {
        return useVocablesShuffleTypeForShowingListsAndSheetsToo;
    }

    /**
     * Legt fest, ob die Sortierung von Vokabellisten auch zum Anzeigen der Listen oder Blätter
     * verwendet werden soll.
     */
    public void setUseVocablesShuffleTypeForShowingListsAndSheetsToo(
            boolean useVocablesShuffleTypeForShowingListsAndSheetsToo) {
        this.useVocablesShuffleTypeForShowingListsAndSheetsToo =
                useVocablesShuffleTypeForShowingListsAndSheetsToo;
    }

    /**
     * Gibt an, ob der Anfang der Beschreibung der Vokabularien ("B1_K04_1_" oder "X_") ausgeblendet
     * werden soll.
     */
    public boolean isHideStartOfVocabularyDescription() {
        return hideStartOfVocabularyDescription;
    }

    /**
     * Legt fest, ob der Anfang der Beschreibung der Vokabularien ("B1_K04_1_" oder "X_")
     * ausgeblendet werden soll.
     */
    public void setHideStartOfVocabularyDescription(boolean hideStartOfVocabularyDescription) {
        this.hideStartOfVocabularyDescription = hideStartOfVocabularyDescription;
    }

    /**
     * Gibt an, ob zum Betrachten der Vokabeln die Reihenfolge zum Abfragen verwendet werden soll.
     */
    public boolean isUseTestOrderForViewingVocables() {
        return useTestOrderForViewingVocables;
    }

    /**
     * Legt fest, ob zum Betrachten der Vokabeln die Reihenfolge zum Abfragen verwendet werden
     * soll.
     */
    public void setUseTestOrderForViewingVocables(boolean useTestOrderForViewingVocables) {
        this.useTestOrderForViewingVocables = useTestOrderForViewingVocables;
    }

    /**
     * Getter für den Index des zuletzt betrachteten Reiters (Vokabularien / eigene Listen) in der
     * Oberfläche.
     */
    public int getLastShownMainTabIndex() {
        return lastShownMainTabIndex;
    }

    /**
     * Setter für den Index des zuletzt betrachteten Reiters (Vokabularien / eigene Listen) in der
     * Oberfläche.
     */
    public void setLastShownMainTabIndex(int lastShownMainTabIndex) {
        this.lastShownMainTabIndex = lastShownMainTabIndex;
    }

    /**
     * Getter für den Index des zuletzt betrachteten Kategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public int getLastShownVocabularyCategoryTabIndex() {
        return lastShownVocabularyCategoryTabIndex;
    }

    /**
     * Setter für den Index des zuletzt betrachteten Kategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public void setLastShownVocabularyCategoryTabIndex(int lastShownVocabularyCategoryTabIndex) {
        this.lastShownVocabularyCategoryTabIndex = lastShownVocabularyCategoryTabIndex;
    }

    /**
     * Getter für den Index des zuletzt betrachteten Kategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public int getLastShownOwnListCategoryTabIndex() {
        return lastShownOwnListCategoryTabIndex;
    }

    /**
     * Setter für den Index des zuletzt betrachteten Kategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public void setLastShownOwnListCategoryTabIndex(int lastShownOwnListCategoryTabIndex) {
        this.lastShownOwnListCategoryTabIndex = lastShownOwnListCategoryTabIndex;
    }

    /**
     * Getter für den Index des zuletzt betrachteten Unterkategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public int getLastShownVocabularySubCategoryTabIndex() {
        return lastShownVocabularySubCategoryTabIndex;
    }

    /**
     * Setter für den Index des zuletzt betrachteten Unterkategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public void setLastShownVocabularySubCategoryTabIndex(
            int lastShownVocabularySubCategoryTabIndex) {
        this.lastShownVocabularySubCategoryTabIndex = lastShownVocabularySubCategoryTabIndex;
    }

    /**
     * Getter für den Index des zuletzt betrachteten Unterkategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public int getLastShownOwnListSubCategoryTabIndex() {
        return lastShownOwnListSubCategoryTabIndex;
    }

    /**
     * Setter für den Index des zuletzt betrachteten Unterkategorie-Reiters (Vokabularien / eigene
     * Listen) in der Oberfläche.
     */
    public void setLastShownOwnListSubCategoryTabIndex(int lastShownOwnListSubCategoryTabIndex) {
        this.lastShownOwnListSubCategoryTabIndex = lastShownOwnListSubCategoryTabIndex;
    }

    /**
     * Gibt an, ob (ggf. neben der Farbe) der Erfolg bei den letzten zehn Abfragen der Vokabeln
     * eines Vokabulars als Prozentzahl mit angezeigt werden soll.
     */
    public boolean isShowSuccessPercentInVocabularyBar() {
        return showSuccessPercentInVocabularyBar;
    }

    /**
     * Legt fest, ob (ggf. neben der Farbe) der Erfolg bei den letzten zehn Abfragen der Vokabeln
     * eines Vokabulars als Prozentzahl mit angezeigt werden soll.
     */
    public void setShowSuccessPercentInVocabularyBar(boolean showSuccessPercentInVocabularyBar) {
        this.showSuccessPercentInVocabularyBar = showSuccessPercentInVocabularyBar;
    }

    /** Getter für die Anzahl der Spalten mit den Vokabularien. */
    public int getNumberOfVocabularyBarColumns() {
        return numberOfVocabularyBarColumns;
    }

    /** Setter für die Anzahl der Spalten mit den Vokabularien. */
    public void setNumberOfVocabularyBarColumns(int numberOfVocabularyBarColumns) {
        this.numberOfVocabularyBarColumns = numberOfVocabularyBarColumns;
    }

    /** Getter für das zuletzt zum Im- oder Exportieren von Vokabellisten verwendete Verzeichnis. */
    public String getLastUsedOwnListImportExportDirectory() {
        return lastUsedOwnListImportExportDirectory;
    }

    /** Setter für das zuletzt zum Im- oder Exportieren von Vokabellisten verwendete Verzeichnis. */
    public void setLastUsedOwnListImportExportDirectory(
            String lastUsedOwnListImportExportDirectory) {
        this.lastUsedOwnListImportExportDirectory = lastUsedOwnListImportExportDirectory;
    }

    /**
     * Getter für die Anzahl an Datensätzen pro Seite im Editor, in welchem eine eigene Listen von
     * Vokabeln bearbeitet wird.
     */
    public int getNumberOfDatasetsPerPageOfOwnListEditor() {
        return numberOfDatasetsPerPageOfOwnListEditor;
    }

    /**
     * Setter für die Anzahl an Datensätzen pro Seite im Editor, in welchem eine eigene Listen von
     * Vokabeln bearbeitet wird.
     */
    public void setNumberOfDatasetsPerPageOfOwnListEditor(
            int numberOfDatasetsPerPageOfOwnListEditor) {
        this.numberOfDatasetsPerPageOfOwnListEditor = numberOfDatasetsPerPageOfOwnListEditor;
    }

    /**
     * Getter für die Anzahl an Spalten mit den auf einer Seite angezeigten Daten im Editor, in
     * welchem eine eigene Listen von Vokabeln bearbeitet wird.
     */
    public int getNumberOfColumnsOfOwnListEditor() {
        return numberOfColumnsOfOwnListEditor;
    }

    /**
     * Setter für die Anzahl an Spalten mit den auf einer Seite angezeigten Daten im Editor, in
     * welchem eine eigene Listen von Vokabeln bearbeitet wird.
     */
    public void setNumberOfColumnsOfOwnListEditor(int numberOfColumnsOfOwnListEditor) {
        this.numberOfColumnsOfOwnListEditor = numberOfColumnsOfOwnListEditor;
    }

    /**
     * Getter für die Position des Editors, in welchem eine eigene Listen von Vokabeln bearbeitet
     * wird.
     */
    public Point getLocationOfOwnListEditor() {
        return locationOfOwnListEditor;
    }

    /**
     * Setter für die Position des Editors, in welchem eine eigene Listen von Vokabeln bearbeitet
     * wird.
     */
    public void setLocationOfOwnListEditor(Point locationOfOwnListEditor) {
        this.locationOfOwnListEditor = locationOfOwnListEditor;
    }

    /**
     * Getter für die Größe des Editors, in welchem eine eigene Listen von Vokabeln bearbeitet
     * wird.
     */
    public Dimension getSizeOfOwnListEditor() {
        return sizeOfOwnListEditor;
    }

    /**
     * Setter für die Größe des Editors, in welchem eine eigene Listen von Vokabeln bearbeitet
     * wird.
     */
    public void setSizeOfOwnListEditor(Dimension sizeOfOwnListEditor) {
        this.sizeOfOwnListEditor = sizeOfOwnListEditor;
    }

    /**
     * Gibt an, ob Vokabeln aus der Liste rechts im Suchbereich versteckt werden sollen. Falls
     * nicht, werden sie nur anders dargestellt.
     */
    public boolean isHideVocablesFromListInSearchAreaInOwnListEditor() {
        return hideVocablesFromListInSearchAreaInOwnListEditor;
    }

    /**
     * Legt fest, ob Vokabeln aus der Liste rechts im Suchbereich versteckt werden sollen. Falls
     * nicht, werden sie nur anders dargestellt.
     */
    public void setHideVocablesFromListInSearchAreaInOwnListEditor(
            boolean hideVocablesFromListInSearchAreaInOwnListEditor) {
        this.hideVocablesFromListInSearchAreaInOwnListEditor =
                hideVocablesFromListInSearchAreaInOwnListEditor;
    }

    /** Gibt an, ob das Programm bei fehlenden MP3-Dateien gestartet werden soll. */
    public boolean isAllowMissingMp3() {
        return allowMissingMp3;
    }

    /** Legt fest, ob das Programm bei fehlenden MP3-Dateien gestartet werden soll. */
    public void setAllowMissingMp3(boolean allowMissingMp3) {
        this.allowMissingMp3 = allowMissingMp3;
    }

    /** Gibt an, ob das Programm beim Start auf fehlende MP3-Dateien hinweisen soll. */
    public boolean isReportMissingMp3() {
        return reportMissingMp3;
    }

    /** Legt fest, ob das Programm beim Start auf fehlende MP3-Dateien hinweisen soll. */
    public void setReportMissingMp3(boolean reportMissingMp3) {
        this.reportMissingMp3 = reportMissingMp3;
    }

    /** Getter für die Anzahl an Spalten in der Ansicht als Vokabelblatt. */
    public int getNumberOfColumnsOfBlattDarstellung() {
        return numberOfColumnsOfBlattDarstellung;
    }

    /** Setter für die Anzahl an Spalten in der Ansicht als Vokabelblatt. */
    public void setNumberOfColumnsOfBlattDarstellung(int numberOfColumnsOfBlattDarstellung) {
        this.numberOfColumnsOfBlattDarstellung = numberOfColumnsOfBlattDarstellung;
    }

    /**
     * Gibt an, ob beim Start des Programms über doppelte MP3-Dateien mit dem gleichen Namen
     * informiert werden soll. Diese gibt es teilweise aus berechtigten Gründen, aber es kann auch
     * auf Fehler hinweisen.
     */
    public boolean isInformAboutDoubleMp3AtStartup() {
        return informAboutDoubleMp3AtStartup;
    }

    /**
     * Legt fest, ob beim Start des Programms über doppelte MP3-Dateien mit dem gleichen Namen
     * informiert werden soll. Diese gibt es teilweise aus berechtigten Gründen, aber es kann auch
     * auf Fehler hinweisen.
     */
    public void setInformAboutDoubleMp3AtStartup(boolean informAboutDoubleMp3AtStartup) {
        this.informAboutDoubleMp3AtStartup = informAboutDoubleMp3AtStartup;
    }

    /**
     * Gibt an, ob bei der Übersetzung von Deutsch nach Japanisch nur der erste deutsche Begriff,
     * welcher in der Vokabel unter translations hinterlegt ist, abgefragt wird. Anderenfalls wird
     * zufällig irgendeiner der deutschen Begriffe abgefragt.
     */
    public boolean isUseOnlyFirstGermanTermWhenTestingVocableGermanToJapanese() {
        return useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese;
    }

    /**
     * Legt fest, ob bei der Übersetzung von Deutsch nach Japanisch nur der erste deutsche Begriff,
     * welcher in der Vokabel unter translations hinterlegt ist, abgefragt wird. Anderenfalls wird
     * zufällig irgendeiner der deutschen Begriffe abgefragt.
     */
    public void setUseOnlyFirstGermanTermWhenTestingVocableGermanToJapanese(
            boolean useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese) {
        this.useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese =
                useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese;
    }

    /**
     * Getter für die Richtung, in die übersetzt wird (Japanisch - Deutsch oder Deutsch -
     * Japanisch).
     */
    public TranslationDirection getTranslationDirection() {
        return translationDirection;
    }

    /**
     * Setter für die Richtung, in die übersetzt wird (Japanisch - Deutsch oder Deutsch -
     * Japanisch).
     */
    public void setTranslationDirection(TranslationDirection translationDirection) {
        this.translationDirection = translationDirection;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Kanji mit angezeigt werden soll.
     */
    public boolean isShowKanjiWhenTestingVocable() {
        return showKanjiWhenTestingVocable;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Darstellung in Kanji mit angezeigt werden soll.
     */
    public void setShowKanjiWhenTestingVocable(boolean showKanjiWhenTestingVocable) {
        this.showKanjiWhenTestingVocable = showKanjiWhenTestingVocable;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Kanji weggelassen werden soll,
     * falls die Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    public boolean isHideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana() {
        return hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Darstellung in Kanji weggelassen werden soll,
     * falls die Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    public void setHideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana(
            boolean hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana) {
        this.hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana =
                hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Romaji weggelassen werden soll,
     * falls die Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    public boolean isHideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana() {
        return hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Darstellung in Romaji weggelassen werden soll,
     * falls die Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    public void setHideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana(
            boolean hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana) {
        this.hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana =
                hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Aussprache weggelassen werden soll, falls die
     * Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    public boolean isHidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana() {
        return hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Aussprache weggelassen werden soll, falls die
     * Darstellung in Kana ausschließlich Hiragana-Zeichen enthält.
     */
    public void setHidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana(
            boolean hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana) {
        this.hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana =
                hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana;
    }

    /** Gibt an, ob der Button zum Anzeigen der Details in der Einzeldarstellung angezeigt wird. */
    public boolean isShowDetailsButtonInEinzeldarstellung() {
        return showDetailsButtonInEinzeldarstellung;
    }

    /**
     * Legt fest, ob der Button zum Anzeigen der Details in der Einzeldarstellung angezeigt wird.
     */
    public void setShowDetailsButtonInEinzeldarstellung(
            boolean showDetailsButtonInEinzeldarstellung) {
        this.showDetailsButtonInEinzeldarstellung = showDetailsButtonInEinzeldarstellung;
    }

    /**
     * Gibt an, ob der Button zum Anzeigen der Details in der Darstellung als Vokabelblatt
     * angezeigt wird.
     */
    public boolean isShowDetailsButtonInVokabelblattDarstellung() {
        return showDetailsButtonInVokabelblattDarstellung;
    }

    /**
     * Legt fest, ob der Button zum Anzeigen der Details in der Darstellung als Vokabelblatt
     * angezeigt wird.
     */
    public void setShowDetailsButtonInVokabelblattDarstellung(
            boolean showDetailsButtonInVokabelblattDarstellung) {
        this.showDetailsButtonInVokabelblattDarstellung =
                showDetailsButtonInVokabelblattDarstellung;
    }

    /**
     * Gibt an, ob bei Vokabularien und Listen aus der VHS weniger vom Anfang der Beschreibung der
     * Vokabularien (entfernt "VHS_A1.1_" statt "VHS_A1.1_01_01_") ausgeblendet werden soll.
     */
    public boolean isHideLessStartOfVocabularyDescriptionForVhs() {
        return hideLessStartOfVocabularyDescriptionForVhs;
    }

    /**
     * Legt fest, ob bei Vokabularien und Listen aus der VHS weniger vom Anfang der Beschreibung
     * der Vokabularien (entfernt "VHS_A1.1_" statt "VHS_A1.1_01_01_") ausgeblendet werden soll.
     */
    public void setHideLessStartOfVocabularyDescriptionForVhs(
            boolean hideLessStartOfVocabularyDescriptionForVhs) {
        this.hideLessStartOfVocabularyDescriptionForVhs =
                hideLessStartOfVocabularyDescriptionForVhs;
    }

    /** Gibt an, ob beim Start des Programms über gleiche Kana und Kanji informiert werden soll. */
    public boolean isInformAboutEqualKanaAndKanjiAtStartup() {
        return informAboutEqualKanaAndKanjiAtStartup;
    }

    /**
     * Legt fest, ob beim Start des Programms über gleiche Kana und Kanji informiert werden soll.
     */
    public void setInformAboutEqualKanaAndKanjiAtStartup(
            boolean informAboutEqualKanaAndKanjiAtStartup) {
        this.informAboutEqualKanaAndKanjiAtStartup = informAboutEqualKanaAndKanjiAtStartup;
    }

    /**
     * Gibt an, ob beim Abfragen der Vokabeln die Darstellung in Kana mit angezeigt werden soll.
     */
    public boolean isShowKanaWhenTestingVocable() {
        return showKanaWhenTestingVocable;
    }

    /**
     * Legt fest, ob beim Abfragen der Vokabeln die Darstellung in Kana mit angezeigt werden soll.
     */
    public void setShowKanaWhenTestingVocable(boolean showKanaWhenTestingVocable) {
        this.showKanaWhenTestingVocable = showKanaWhenTestingVocable;
    }

    /**
     * Gibt an, ob beim Start des Programms auf eine aktuellere Version der eigenen Listen geprüft
     * werden soll.
     */
    public boolean isCheckNewOwnListsVersionAtStart() {
        return checkNewOwnListsVersionAtStart;
    }

    /**
     * Legt fest, ob beim Start des Programms auf eine aktuellere Version der eigenen Listen
     * geprüft werden soll.
     */
    public void setCheckNewOwnListsVersionAtStart(boolean checkNewOwnListsVersionAtStart) {
        this.checkNewOwnListsVersionAtStart = checkNewOwnListsVersionAtStart;
    }

    /**
     * Gibt an, ob die Kana, wenn auch Kanji dargestellt werden, beim Abfragen genauso groß wie die
     * Kanji dargestellt werden sollen. Anderenfalls werden sie etwas kleiner dargestellt.
     */
    public boolean isShowKanaInKanjiSizeWhenTestingVocable() {
        return showKanaInKanjiSizeWhenTestingVocable;
    }

    /**
     * Legt fest, ob die Kana, wenn auch Kanji dargestellt werden, beim Abfragen genauso groß wie
     * die Kanji dargestellt werden sollen. Anderenfalls werden sie etwas kleiner dargestellt.
     */
    public void setShowKanaInKanjiSizeWhenTestingVocable(
            boolean showKanaInKanjiSizeWhenTestingVocable) {
        this.showKanaInKanjiSizeWhenTestingVocable = showKanaInKanjiSizeWhenTestingVocable;
    }

    /** Getter für die Art, ob auf das Abschicken einer leeren Übersetzung reagiert werden soll. */
    public EmptyTranslationsAcceptance getEmptyTranslationsAcceptance() {
        return emptyTranslationsAcceptance;
    }

    /** Setter für die Art, ob auf das Abschicken einer leeren Übersetzung reagiert werden soll. */
    public void setEmptyTranslationsAcceptance(
            EmptyTranslationsAcceptance emptyTranslationsAcceptance) {
        this.emptyTranslationsAcceptance = emptyTranslationsAcceptance;
    }

    /**
     * Gibt an, wieviele Tage 'eine Weile' ist, um Vokabeln anzuzeigen oder abzufragen, die 'lange
     * nicht mehr abgefragt' wurden.
     */
    public int getNumberOfDaysForAWhile() {
        return numberOfDaysForAWhile;
    }

    /**
     * Legt fest, wieviele Tage 'eine Weile' ist, um Vokabeln anzuzeigen oder abzufragen, die
     * 'lange nicht mehr abgefragt' wurden.
     */
    public void setNumberOfDaysForAWhile(int numberOfDaysForAWhile) {
        this.numberOfDaysForAWhile = numberOfDaysForAWhile;
    }

    /**
     * Gibt an wieviele Tage seit des ersten Auftretens eine Vokabel noch als 'neu' gilt, um die
     * 'neuesten' Vokabeln abzufragen oder anzuzeigen.
     */
    public int getNumberOfDaysAVocableIsNew() {
        return numberOfDaysAVocableIsNew;
    }

    /**
     * Legt fest wieviele Tage seit des ersten Auftretens eine Vokabel noch als 'neu' gilt, um die
     * 'neuesten' Vokabeln abzufragen oder anzuzeigen.
     */
    public void setNumberOfDaysAVocableIsNew(int numberOfDaysAVocableIsNew) {
        this.numberOfDaysAVocableIsNew = numberOfDaysAVocableIsNew;
    }

    /**
     * Getter für die Liste der elf Farben von schlecht bis gut (0/10, 1/10 bis 10/10 richtige
     * Antworten).
     */
    public List<Color> getColors() {
        return colors;
    }

    /**
     * Setter für die Liste der elf Farben von schlecht bis gut (0/10, 1/10 bis 10/10 richtige
     * Antworten).
     */
    public void setColors(List<Color> colors) {
        if (colors.size() != VocableColors.NUMBER_OF_COLORS) {
            throw new RuntimeException("Es gibt nicht wie erwartet "
                    + VocableColors.NUMBER_OF_COLORS + " Farben, sondern " + colors.size() + ".");
        }

        this.colors = colors;
    }

    /** Getter für den Wert um den die Farben für eine Vordergrundfarbe abgedunkelt werden. */
    public int getDeltaForForegroundColor() {
        return deltaForForegroundColor;
    }

    /** Setter für den Wert um den die Farben für eine Vordergrundfarbe abgedunkelt werden. */
    public void setDeltaForForegroundColor(int deltaForForegroundColor) {
        this.deltaForForegroundColor = deltaForForegroundColor;
    }

    /** Getter für den Wert um den die Farben für eine Hintergrundfarbe aufgehellt werden. */
    public int getDeltaForBackgroundColor() {
        return deltaForBackgroundColor;
    }

    /** Setter für den Wert um den die Farben für eine Hintergrundfarbe aufgehellt werden. */
    public void setDeltaForBackgroundColor(int deltaForBackgroundColor) {
        this.deltaForBackgroundColor = deltaForBackgroundColor;
    }

    /**
     * Getter für die Art wie Farben für die Erzeugung von Vorder- bzw. Hintergrundfarben
     * modifiziert werden.
     */
    public ColorModificationType getColorModificationType() {
        return colorModificationType;
    }

    /**
     * Setter für die Art wie Farben für die Erzeugung von Vorder- bzw. Hintergrundfarben
     * modifiziert werden.
     */
    public void setColorModificationType(ColorModificationType colorModificationType) {
        this.colorModificationType = colorModificationType;
    }

    /**
     * Gibt an, ob im Dialog zur Bearbeitung einer eigenen Liste ein variabler Trenner zwischen dem
     * linken Teil mit der Auswahl der Vokabeln und dem rechten Teil mit der Liste verwendet wird.
     */
    public boolean isUseSplitPaneInOwnListEditor() {
        return useSplitPaneInOwnListEditor;
    }

    /**
     * Legt fest, ob im Dialog zur Bearbeitung einer eigenen Liste ein variabler Trenner zwischen
     * dem linken Teil mit der Auswahl der Vokabeln und dem rechten Teil mit der Liste verwendet
     * wird.
     */
    public void setUseSplitPaneInOwnListEditor(boolean useSplitPaneInOwnListEditor) {
        this.useSplitPaneInOwnListEditor = useSplitPaneInOwnListEditor;
    }

    /** Getter für die Farbe für normalen Text bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorStandard() {
        return vocableColorStandard;
    }

    /** Setter für die Farbe für normalen Text bei der Darstellung einer Vokabel. */
    public void setVocableColorStandard(Color vocableColorStandard) {
        this.vocableColorStandard = vocableColorStandard;
    }

    /** Getter für die Farbe für Kanji bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorKanji() {
        return vocableColorKanji;
    }

    /** Setter für die Farbe für Kanji bei der Darstellung einer Vokabel. */
    public void setVocableColorKanji(Color vocableColorKanji) {
        this.vocableColorKanji = vocableColorKanji;
    }

    /** Getter für die Farbe für Kana bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorKana() {
        return vocableColorKana;
    }

    /** Setter für die Farbe für Kana bei der Darstellung einer Vokabel. */
    public void setVocableColorKana(Color vocableColorKana) {
        this.vocableColorKana = vocableColorKana;
    }

    /** Getter für die Farbe für die Romaji bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorRomaji() {
        return vocableColorRomaji;
    }

    /** Setter für die Farbe für die Romaji bei der Darstellung einer Vokabel. */
    public void setVocableColorRomaji(Color vocableColorRomaji) {
        this.vocableColorRomaji = vocableColorRomaji;
    }

    /** Getter für die Farbe für die Aussprache bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorPronunciation() {
        return vocableColorPronunciation;
    }

    /** Setter für die Farbe für die Aussprache bei der Darstellung einer Vokabel. */
    public void setVocableColorPronunciation(Color vocableColorPronunciation) {
        this.vocableColorPronunciation = vocableColorPronunciation;
    }

    /** Getter für die Farbe für die Übersetzung bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorTranslation() {
        return vocableColorTranslation;
    }

    /** Setter für die Farbe für die Übersetzung bei der Darstellung einer Vokabel. */
    public void setVocableColorTranslation(Color vocableColorTranslation) {
        this.vocableColorTranslation = vocableColorTranslation;
    }

    /** Getter für die Farbe für den Kommentar bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorComment() {
        return vocableColorComment;
    }

    /** Setter für die Farbe für den Kommentar bei der Darstellung einer Vokabel. */
    public void setVocableColorComment(Color vocableColorComment) {
        this.vocableColorComment = vocableColorComment;
    }

    /** Getter für die Farbe für einen Suchbegriff bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorSearchWords() {
        return vocableColorSearchWords;
    }

    /** Setter für die Farbe für einen Suchbegriff bei der Darstellung einer Vokabel. */
    public void setVocableColorSearchWords(Color vocableColorSearchWords) {
        this.vocableColorSearchWords = vocableColorSearchWords;
    }

    /** Getter für die Farbe für eine Wortart bei der Darstellung einer Vokabel. */
    @Override
    public Color getVocableColorPartOfSpeach() {
        return vocableColorPartOfSpeach;
    }

    /** Setter für die Farbe für eine Wortart bei der Darstellung einer Vokabel. */
    public void setVocableColorPartOfSpeach(Color vocableColorPartOfSpeach) {
        this.vocableColorPartOfSpeach = vocableColorPartOfSpeach;
    }

    /**
     * Gibt an, ob beim Startup die bis zu diesem Schritt bereits benötigte Laufzeit vor den
     * Meldungen ausgegeben werden soll.
     */
    public boolean isShowTimestampsInStartup() {
        return showTimestampsInStartup;
    }

    /**
     * Legt fest, ob beim Startup die bis zu diesem Schritt bereits benötigte Laufzeit vor den
     * Meldungen ausgegeben werden soll.
     */
    public void setShowTimestampsInStartup(boolean showTimestampsInStartup) {
        this.showTimestampsInStartup = showTimestampsInStartup;
    }

    /**
     * Gibt an, ob das Verzeichnis für die Übersetzung Deutsch nach Japanisch beim Start aufgebaut
     * werden soll. Dies dauert einige Zeit, die der Start länger dauert. Im anderen Fall wird
     * dieses Verzeichnis erst aufgebaut, wenn es das erste Mal benötigt wird, also wenn die erste
     * Vokabel von Deutsch nach Japanisch übersetzt wird.
     *
     * Übersetzt man oft auch von Deutsch nach Japanisch, sollte man es beim Startup erzeugen,
     * anderenfalls ist es besser, es erst dann zu erzeugen, wenn man es braucht, weil man es
     * meistens nicht braucht und so der Vokabel-Trainer schneller startet.
     */
    public boolean isCreateGermanJapaneseTranslationAtStartup() {
        return createGermanJapaneseTranslationAtStartup;
    }

    /**
     * Legt fest, ob das Verzeichnis für die Übersetzung Deutsch nach Japanisch beim Start
     * aufgebaut werden soll. Dies dauert einige Zeit, die der Start länger dauert. Im anderen Fall
     * wird dieses Verzeichnis erst aufgebaut, wenn es das erste Mal benötigt wird, also wenn die
     * erste Vokabel von Deutsch nach Japanisch übersetzt wird.
     *
     * Übersetzt man oft auch von Deutsch nach Japanisch, sollte man es beim Startup erzeugen,
     * anderenfalls ist es besser, es erst dann zu erzeugen, wenn man es braucht, weil man es
     * meistens nicht braucht und so der Vokabel-Trainer schneller startet.
     */
    public void setCreateGermanJapaneseTranslationAtStartup(
            boolean createGermanJapaneseTranslationAtStartup) {
        this.createGermanJapaneseTranslationAtStartup = createGermanJapaneseTranslationAtStartup;
    }

    /** Getter für das zuletzt zum Auspacken verwendete 7Zip-Executable. */
    public String getLastUsed7ZipDirectory() {
        return lastUsed7ZipDirectory;
    }

    /** Setter für das zuletzt zum Auspacken verwendete 7Zip-Executable. */
    public void setLastUsed7ZipDirectory(String lastUsed7ZipDirectory) {
        this.lastUsed7ZipDirectory = lastUsed7ZipDirectory;
    }

    /**
     * Getter für den zuletzt ausgewählten Text der Combobox zur Auswahl der abzufragenden Menge an
     * Kanji im KanjiTestParameterChooser.
     */
    public String getLastSelectedKanjiSetTextInCombobox() {
        return lastSelectedKanjiSetTextInCombobox;
    }

    /**
     * Setter für den zuletzt ausgewählten Text der Combobox zur Auswahl der abzufragenden Menge an
     * Kanji im KanjiTestParameterChooser.
     */
    public void setLastSelectedKanjiSetTextInCombobox(String lastSelectedKanjiSetTextInCombobox) {
        this.lastSelectedKanjiSetTextInCombobox = lastSelectedKanjiSetTextInCombobox;
    }

    /**
     * Gibt an, ob beim Start geprüft werden soll, ob "e" oder "u" in der Aussprache der Vokabeln
     * vorkommt. Dies ist nur sinnvoll für jemanden, der die Vokabeln direkt bearbeitet. Sinn
     * dahinter ist, dass ich diese Vokabel in der Aussprache einheitlich als "ä" und "ü"
     * darstellen möchte.
     */
    public boolean isInformAboutEOrUInVocableFieldAusspracheAtStartup() {
        return informAboutEOrUInVocableFieldAusspracheAtStartup;
    }

    /**
     * Legt fest, ob beim Start geprüft werden soll, ob "e" oder "u" in der Aussprache der Vokabeln
     * vorkommt. Dies ist nur sinnvoll für jemanden, der die Vokabeln direkt bearbeitet. Sinn
     * dahinter ist, dass ich diese Vokabel in der Aussprache einheitlich als "ä" und "ü"
     * darstellen möchte.
     */
    public void setInformAboutEOrUInVocableFieldAusspracheAtStartup(
            boolean informAboutEOrUInVocableFieldAusspracheAtStartup) {
        this.informAboutEOrUInVocableFieldAusspracheAtStartup =
                informAboutEOrUInVocableFieldAusspracheAtStartup;
    }

    /**
     * Getter für die Anzahl der in der Historie im Listen-Commander links und rechts sowie beim
     * Hinzufügen von Vokabeln zu eigenen Listen über den Detail-Dialog angezeigten besuchten
     * Listen.
     */
    public int getNumberOfRememberedOwnLists() {
        return numberOfRememberedOwnLists;
    }

    /**
     * Setter für die Anzahl der in der Historie im Listen-Commander links und rechts sowie beim
     * Hinzufügen von Vokabeln zu eigenen Listen über den Detail-Dialog angezeigten besuchten
     * Listen.
     */
    public void setNumberOfRememberedOwnLists(int numberOfRememberedOwnLists) {
        this.numberOfRememberedOwnLists = numberOfRememberedOwnLists;
        historicalLeftOwnListsInListCommander.setNumberOfRememberedOwnLists(
                numberOfRememberedOwnLists);
        historicalRightOwnListsInListCommander.setNumberOfRememberedOwnLists(
                numberOfRememberedOwnLists);
    }

    /**
     * Getter für die zuletzt auf der linken Seite des List-Commanders genutzten eigenen Listen.
     */
    public HistoricalOwnListPersistanceDataList getHistoricalLeftOwnListsInListCommander() {
        return historicalLeftOwnListsInListCommander;
    }

    /**
     * Setter für die zuletzt auf der linken Seite des List-Commanders genutzten eigenen Listen.
     */
    public void setHistoricalLeftOwnListsInListCommander(
            HistoricalOwnListPersistanceDataList historicalLeftOwnListsInListCommander) {
        this.historicalLeftOwnListsInListCommander = historicalLeftOwnListsInListCommander;
    }

    /**
     * Getter für die zuletzt auf der rechten Seite des List-Commanders genutzten eigenen Listen.
     */
    public HistoricalOwnListPersistanceDataList getHistoricalRightOwnListsInListCommander() {
        return historicalRightOwnListsInListCommander;
    }

    /**
     * Setter für die zuletzt auf der rechten Seite des List-Commanders genutzten eigenen Listen.
     */
    public void setHistoricalRightOwnListsInListCommander(
            HistoricalOwnListPersistanceDataList historicalRightOwnListsInListCommander) {
        this.historicalRightOwnListsInListCommander = historicalRightOwnListsInListCommander;
    }

    /** Getter für die zuletzt außerhalb des List-Commanders genutzten eigenen Listen. */
    public HistoricalOwnListPersistanceDataList getHistoricalOwnLists() {
        return historicalOwnLists;
    }

    /** Setter für die zuletzt außerhalb des List-Commanders genutzten eigenen Listen. */
    public void setHistoricalOwnLists(
            HistoricalOwnListPersistanceDataList historicalOwnLists) {
        this.historicalOwnLists = historicalOwnLists;
    }

    /**
     * Gibt an, ob in der Einzeldarstellung von Vokabeln und im Dialog zur Bewertung einer
     * Übersetzung alle Übersetzungen angezeigt werden.
     *
     * Dies kann bei vielen Übersetzungen zur Überlänge führen, so dass die Dialoge nach unten aus
     * dem Bildschirm ragen.
     */
    public boolean isShowAllTranslations() {
        return showAllTranslations;
    }

    /**
     * Legt fest, ob in der Einzeldarstellung von Vokabeln und im Dialog zur Bewertung einer
     * Übersetzung alle Übersetzungen angezeigt werden.
     *
     * Dies kann bei vielen Übersetzungen zur Überlänge führen, so dass die Dialoge nach unten aus
     * dem Bildschirm ragen.
     */
    public void setShowAllTranslations(boolean showAllTranslations) {
        this.showAllTranslations = showAllTranslations;
    }

    /**
     * Gibt an, wieviele Übersetzungen in der Einzeldarstellung von Vokabeln und im Dialog zur
     * Bewertung einer Übersetzung angezeigt werden, falls denn der Benutzer nicht alle
     * Übersetzungen anzeigen möchte (siehe showAllTranslations).
     *
     * Die Anzeige aller oder zu vieler Übersetzungen kann bei vielen Übersetzungen zur Überlänge
     * führen, so dass die Dialoge nach unten aus dem Bildschirm ragen. Mit dieser Einstellung kann
     * man abhängig von der Monitorauflösung eine passende Wahl treffen.
     */
    public int getNumberOfShownTranslations() {
        return numberOfShownTranslations;
    }

    /**
     * Legt fest, wieviele Übersetzungen in der Einzeldarstellung von Vokabeln und im Dialog zur
     * Bewertung einer Übersetzung angezeigt werden, falls denn der Benutzer nicht alle
     * Übersetzungen anzeigen möchte (siehe showAllTranslations).
     *
     * Die Anzeige aller oder zu vieler Übersetzungen kann bei vielen Übersetzungen zur Überlänge
     * führen, so dass die Dialoge nach unten aus dem Bildschirm ragen. Mit dieser Einstellung kann
     * man abhängig von der Monitorauflösung eine passende Wahl treffen.
     */
    public void setNumberOfShownTranslations(int numberOfShownTranslations) {
        this.numberOfShownTranslations = numberOfShownTranslations;
    }

    /**
     * Gibt an, ob die Anzahl der Listen, in der die Vokabel ist, zusammen mit einem Hinweis auf
     * Lernlisten, in der Listendarstellung eines Vokabulars oder einer eigenen Liste angezeigt
     * werden soll.
     */
    public boolean isShowNumberOfListsInLister() {
        return showNumberOfListsInLister;
    }

    /**
     * Legt fest, ob die Anzahl der Listen, in der die Vokabel ist, zusammen mit einem Hinweis auf
     * Lernlisten, in der Listendarstellung eines Vokabulars oder einer eigenen Liste angezeigt
     * werden soll.
     */
    public void setShowNumberOfListsInLister(boolean showNumberOfListsInLister) {
        this.showNumberOfListsInLister = showNumberOfListsInLister;
    }

    /**
     * Gibt an, ob die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum
     * Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen der bekannten
     * Verb- bzw. Adjektivformen, auch im Dialog mit der Einzeldarstellung angezeigt werden sollen.
     */
    public boolean isShowDetailBottomButonsInEinzeldarstellungToo() {
        return showDetailBottomButonsInEinzeldarstellungToo;
    }

    /**
     * Legt fest, ob die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum
     * Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen der bekannten
     * Verb- bzw. Adjektivformen, auch im Dialog mit der Einzeldarstellung angezeigt werden sollen.
     */
    public void setShowDetailBottomButonsInEinzeldarstellungToo(
            boolean showDetailBottomButonsInEinzeldarstellungToo) {
        this.showDetailBottomButonsInEinzeldarstellungToo =
                showDetailBottomButonsInEinzeldarstellungToo;
    }

    /**
     * Gibt an, ob die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum
     * Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen der bekannten
     * Verb- bzw. Adjektivformen, auch im Dialog mit der Bewertung einer einzelnen Abfrage einer
     * Vokabel angezeigt werden sollen.
     */
    public boolean isShowDetailBottomButonsInTranslationEvaluationToo() {
        return showDetailBottomButonsInTranslationEvaluationToo;
    }

    /**
     * Legt fest, ob die Buttons zum Anzeigen der eigenen Listen, welche die Vokabel enthalten, zum
     * Hinzufügen der Vokabel zu einer eigenen Liste und ggf. dem Button zum Anzeigen der bekannten
     * Verb- bzw. Adjektivformen, auch im Dialog mit der Bewertung einer einzelnen Abfrage einer
     * Vokabel angezeigt werden sollen.
     */
    public void setShowDetailBottomButonsInTranslationEvaluationToo(
            boolean showDetailBottomButonsInTranslationEvaluationToo) {
        this.showDetailBottomButonsInTranslationEvaluationToo =
                showDetailBottomButonsInTranslationEvaluationToo;
    }

    /** Gibt an, ob beim Abfragen der Vokabeln diese automatisch vorgelesen werden sollen. */
    public boolean isPlayMP3WhenTestingVocables() {
        return playMP3WhenTestingVocables;
    }

    /** Legt fest, ob beim Abfragen der Vokabeln diese automatisch vorgelesen werden sollen. */
    public void setPlayMP3WhenTestingVocables(boolean playMP3WhenTestingVocables) {
        this.playMP3WhenTestingVocables = playMP3WhenTestingVocables;
    }

    /**
     * Gibt an, ob beim Abfragen der Kanji werden die Felder der Benutzereingaben im Dialog,
     * welcher Erfolg oder Misserfolg der Angaben zu einem einzelnen Kanji anzeigt, rot oder grün
     * eingefärbt.
     */
    public boolean isColorUserInputAtKanjiTestEvaluationDialog() {
        return colorUserInputAtKanjiTestEvaluationDialog;
    }

    /**
     * Legt fest, ob beim Abfragen der Kanji werden die Felder der Benutzereingaben im Dialog,
     * welcher Erfolg oder Misserfolg der Angaben zu einem einzelnen Kanji anzeigt, rot oder grün
     * eingefärbt.
     */
    public void setColorUserInputAtKanjiTestEvaluationDialog(
            boolean colorUserInputAtKanjiTestEvaluationDialog) {
        this.colorUserInputAtKanjiTestEvaluationDialog = colorUserInputAtKanjiTestEvaluationDialog;
    }

}
