package de.duehl.vocabulary.japanese.data;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;

import static de.duehl.vocabulary.japanese.common.persistence.SessionManager.VOCABLE_TRAINER_DIRECTORY;

/**
 * Diese Klasse steht für eine (Teil-) Menge von Kanji, die der Benutzer definieren kann.
 *
 * @version 1.01     2025-04-11
 * @author Christian Dühl
 */

public class KanjiSet {

    /** Das benutzerspezifische Verzeichnis, in dem die eigenen Listen liegen. */
    public static final String KANJI_SET_DIRECTORY =
            FileHelper.concatPathes(VOCABLE_TRAINER_DIRECTORY, "kanji_mengen");

    /** Die Extension für die Dateien mit den Vokabellisten. */
    public static final String KANJI_SET_EXTENSION = ".kst";


    /** Der Name der Kanji-Menge. */
    private String name;

    /** Die Kanji die zu der Menge gehören. */
    private final List<Kanji> set;

    /**
     * Konstruktor
     *
     * @param name
     *            Der Name der Kanji-Menge.
     */
    public KanjiSet(String name) {
        if (name.isBlank()) {
            throw new RuntimeException("Der Name '" + name + "' ist unzulässig.");
        }

        this.name = name;
        set = new ArrayList<>();
    }

    /** Fügt ein Kanji zur Menge hinzu. */
    public void addKanji(Kanji kanji) {
        if (set.contains(kanji)) {
            throw new RuntimeException("Das Kanji ist bereit enthalten: " + kanji);
        }
        else {
            set.add(kanji);
        }
    }

    /** Fügt ein Kanji zur Menge hinzu. */
    public void addKanji(String kanjiCharakter) {
        Kanji kanji = Kanji.getByCharacter(kanjiCharakter);
        set.add(kanji);
    }

    /** Getter für den Namen der Kanji-Menge. */
    public String getName() {
        return name;
    }

    /** Setter für den Namen der Kanji-Menge. */
    public void setName(String name) {
        this.name = name;
    }

    /** Getter für die Kanji die zu der Menge gehören. */
    public List<Kanji> getSet() {
        return set;
    }

    /** Ermittelt den passenden Dateinamen im passenden Verzeichnis der Benutzers. */
    public String determineFilename() {
        return determineFilename(name);
    }

    /** Ermittelt zum Namen den passenden Dateinamen im passenden Verzeichnis der Benutzers. */
    public static String determineFilename(String name) {
        String bareFilename = determineBareFilename(name);
        String filename = FileHelper.concatPathes(KANJI_SET_DIRECTORY, bareFilename);
        return filename;
    }

    /** Ermittelt zum Namen den passenden Dateinamen ohne PFad. */
    public static String determineBareFilename(String name) {
        String bareFilename = Text.createJavaVariableName(name) + KANJI_SET_EXTENSION;
        return bareFilename;
    }

//    /**
//     * Erzeugt ein KanjiSet aus einer Liste von Kanji und einem Namen.
//     *
//     * @param kanjiList
//     *            Die Liste der hinzuzufügenden Kanji.
//     * @param name
//     *            Der Name der Kanji-Menge.
//     * @return Das erzeugte KanjiSet.
//     */
//    public static KanjiSet createSetFromList(List<Kanji> kanjiList, String name) {
//        KanjiSet kanjiSet = new KanjiSet(name);
//
//        for (Kanji kanji : kanjiList) {
//            kanjiSet.addKanji(kanji);
//        }
//
//        return kanjiSet;
//    }

}
