package de.duehl.vocabulary.japanese.ui.dialog;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.basics.text.NumberString;
import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;
import de.duehl.swing.ui.pages.DatasetsOnPages;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.logic.VocabularyTrainerLogic;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.logic.test.VocableListTesterLogic;
import de.duehl.vocabulary.japanese.logic.view.CompleteVocabularyViewerLogic;
import de.duehl.vocabulary.japanese.ui.VocabularyTrainerGui;
import de.duehl.vocabulary.japanese.ui.components.bars.VocableBar;
import de.duehl.vocabulary.japanese.ui.data.MessageSetter;
import de.duehl.vocabulary.japanese.ui.dialog.detail.VocableWithInternaDialog;
import de.duehl.vocabulary.japanese.ui.filter.VocableFilterInputs;
import de.duehl.vocabulary.japanese.ui.filter.VocableFilterPanel;

/**
 * Diese Klasse stellt den Dialog zur erweiterten Suche nach Vokabeln im Vokabel-Trainer dar.
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

public class ComplexVocableSearchDialog extends ModalDialogBase {

    /** Die Logik des Vokabel-Trainers. */
    private final VocabularyTrainerLogic logic;

    /** Die grafische Oberfläche des Vokabel-Trainers. */
    private final VocabularyTrainerGui gui;

    /** Das Element mit den Filterkriterien für die Vokabeln. */
    private final VocableFilterPanel vocableFilter;

    /** Zur Anzeiger der Statistik. */
    private final JLabel statisticLabel;

    /** Zeigt die Vokabeln die den Filterkriterien entsprechen an. */
    private final DatasetsOnPages<Vocable> pages;

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalKanjiDataRequester kanjiRequester;

    /** Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann. */
    private final MessageSetter messageSetter;

    /**
     * Die Eingaben der Suchkriterien. Diese werden zwischen zwei Aufrufen in der selben Sitzung
     * hier gespeichert.
     */
    private static VocableFilterInputs inputs;

    /**
     * Konstruktor.
     *
     * @param logic
     *            Die Logik des Vokabel-Trainers.
     * @param gui
     *            Die grafische Oberfläche des Vokabel-Trainers.
     * @param ownLists
     *            Die Logik zu den eigenen Vokabellisten.
     * @param kanjiRequester
     *            Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param messageSetter
     *            Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann.
     */
    public ComplexVocableSearchDialog(VocabularyTrainerLogic logic, VocabularyTrainerGui gui,
            OwnLists ownLists, InternalKanjiDataRequester kanjiRequester,
            MessageSetter messageSetter) {
        super(gui.getLocation(), gui.getProgramImage(), "Erweiterte Suche nach Vokabeln");
        addEscapeBehaviour();
        addClosingWindowListener(() -> storeSearchParametersAndCloseDialog());

        this.logic = logic;
        this.gui = gui;
        this.ownLists = ownLists;
        this.kanjiRequester = kanjiRequester;
        this.messageSetter = messageSetter;

        vocableFilter = new VocableFilterPanel(ownLists,
                filteredVocablesList -> reactOnFilteredVocables(filteredVocablesList));

        statisticLabel = new JLabel();

        Options options = ownLists.getOptions();
        int numberOfDatasetsPerSide = options.getNumberOfDatasetsPerPageOfOwnListEditor();
        int numberOfColumns = options.getNumberOfColumnsOfOwnListEditor();
        pages = new DatasetsOnPages<>(logic.collectVocablesOfAllVocabularies(),
                vocable -> createDatasetUi(vocable), numberOfDatasetsPerSide, numberOfColumns);

        if (null == inputs) {
            inputs = new VocableFilterInputs();
        }

        init();
        fillDialog();
    }

    /** Erstellt aus dem übergebenen Datensatz die Anzeige für die Gui. */
    private Component createDatasetUi(Vocable vocable) {
        VocableBar bar = new VocableBar(vocable, () -> showDetailDialog(vocable));
        bar.useButtonAsShowDetails();
        bar.createGui();
        return bar.getPanel();
    }

    private void showDetailDialog(Vocable vocable) {
        VocableWithInternaDialog viewer = new VocableWithInternaDialog(logic.getOptions(),
                logic.getInternalDataRequester(), vocable, ownLists, kanjiRequester, messageSetter,
                getLocation(), getProgramImage());
        viewer.setVisible(true);
    }

    private void init() {
        initVocableFilter();
        initStatistic();
    }

    private void initVocableFilter() {
        vocableFilter.arrangeForComplexVocableSearchDialog();
        vocableFilter.loadInputs(inputs);
    }

    private void initStatistic() {
        updateStatistic(vocableFilter.createFilteredVocablesList());
    }

    private void reactOnFilteredVocables(List<Vocable> filteredVocablesList) {
        updateStatistic(filteredVocablesList);
        if (filteredVocablesList.isEmpty()) {
            /*
             * Damit kommt der ListNavigator nicht klar, weil er dann keinen aktuellen Datensatz
             * zurückgeben kann.
             *
             * Das wäre schön, wenn die Liste dann einfach leer wäre, ist aber ein größerer
             * Eingriff...
             */
            GuiTools.informUser(getWindowAsComponent(), "Keine Vokabeln gefunden",
                    "Die Suchkriterien führen zu einer leeren Liste von Vokabeln, daher wird "
                            + "diese nicht angezeigt.");
        }
        else {
            pages.setOtherDatasets(filteredVocablesList);
        }
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createSearchAndStatisticPart(), BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);
    }

    private Component createSearchAndStatisticPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createSearchAndFoundVocablesPart(), BorderLayout.CENTER);
        panel.add(createStatisticsPart(), BorderLayout.SOUTH);

        return panel;
    }

    private Component createSearchAndFoundVocablesPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createSearchPart(), BorderLayout.WEST);
        panel.add(createFoundVocablesPart(), BorderLayout.CENTER);

        return panel;
    }

    private Component createSearchPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle("Kriterien", panel);

        panel.add(vocableFilter.getPanel(), BorderLayout.NORTH);

        return panel;
    }

    private Component createFoundVocablesPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle("Treffer", panel);

        panel.add(pages.getPanel(), BorderLayout.CENTER);

        return panel;
    }

    private Component createStatisticsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle("Statistik", panel);

        panel.add(statisticLabel, BorderLayout.CENTER);

        return panel;
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(1, 0, 5, 0));

        panel.add(createTestVocablesButton());
        panel.add(createEinzeldarsetllungButton());
        panel.add(createListendarstellungButton());
        panel.add(createVokabelblattdarstellungButton());
        panel.add(createSaveListButton());
        panel.add(createCloseDialogButton());

        return panel;
    }

    private Component createTestVocablesButton() {
        JButton button = new JButton("Vokabeln abfragen");
        button.addActionListener(e -> testVocables());
        return button;
    }

    private void testVocables() {
        List<Vocable> foundVocables = vocableFilter.createFilteredVocablesList();
        if (foundVocables.isEmpty()) {
            informNoVocablesMatchSearchCriteria();
        }
        else {
            storeSearchParametersAndCloseDialog();
            String vocablesTitle = "Vokabeln aus der erweiterten Suche";
            VocableListTesterLogic tester = new VocableListTesterLogic(logic, gui, foundVocables,
                    vocablesTitle, messageSetter);
            tester.test();

            gui.setCorrectForegroundColorOfVocabularyBarsLater();
            gui.setMessageLater("Getestet wurden: " + vocablesTitle);
        }
    }

    private Component createEinzeldarsetllungButton() {
        JButton button = new JButton("Einzeldarstellung");
        button.addActionListener(e -> showAsEinzeldarstellung());
        return button;
    }

    private void showAsEinzeldarstellung() {
        List<Vocable> foundVocables = vocableFilter.createFilteredVocablesList();
        if (foundVocables.isEmpty()) {
            informNoVocablesMatchSearchCriteria();
        }
        else {
            storeSearchParametersAndCloseDialog();
            CompleteVocabularyViewerLogic viewer = new CompleteVocabularyViewerLogic(
                    logic.getOptions(), logic.getInternalDataRequester(), foundVocables,
                    logic.getInternalKanjiDataRequester(), "Gefundene Vokabeln",
                    (LongTimeProcessInformer) gui, ownLists, messageSetter, getLocation(),
                    getProgramImage());
            viewer.view();
            gui.setMessageLater("Die gefundenen Vokabeln wurden angezeigt.");
        }
    }

    private void informNoVocablesMatchSearchCriteria() {
        String message = "Es gibt keine Vokabeln, die auf die Suchkriterien passen.";
        GuiTools.informUser(getWindowAsComponent(), "Hinweis", message);
        gui.setMessageLater(message);
    }

    private Component createListendarstellungButton() {
        JButton button = new JButton("Listendarstellung");
        button.addActionListener(e -> showAsListendarstellung());
        return button;
    }

    private void showAsListendarstellung() {
        List<Vocable> foundVocables = vocableFilter.createFilteredVocablesList();
        if (foundVocables.isEmpty()) {
            informNoVocablesMatchSearchCriteria();
        }
        else {
            storeSearchParametersAndCloseDialog();
            VocabularyListerDialog dialog = new VocabularyListerDialog(logic.getOptions(),
                    logic.getInternalDataRequester(), foundVocables,
                    "Vokabeln aus der erweiterten Suche", logic.getInternalKanjiDataRequester(),
                    (LongTimeProcessInformer) gui, ownLists, messageSetter, getLocation(),
                    getProgramImage());
            dialog.setVisible(true);
            gui.setMessageLater(
                    "Die Vokabeln aus der erweiterten Suche wurden als Liste angezeigt.");
        }
    }

    private Component createVokabelblattdarstellungButton() {
        JButton button = new JButton("Darstellung als Vokabelblatt");
        button.addActionListener(e -> showAsVokabelblattdarstellung());
        return button;
    }

    private void showAsVokabelblattdarstellung() {
        List<Vocable> foundVocables = vocableFilter.createFilteredVocablesList();
        if (foundVocables.isEmpty()) {
            informNoVocablesMatchSearchCriteria();
        }
        else {
            storeSearchParametersAndCloseDialog();
            gui.sheetWithVocables(foundVocables, "Vokabeln aus der erweiterten Suche");
            gui.setMessageLater(
                    "Die Vokabeln aus der erweiterten Suche wurden als Blatt angezeigt.");
        }
    }

    private Component createSaveListButton() {
        JButton button = new JButton("Gefundene Vokabeln als Liste speichern");
        button.addActionListener(e -> saveList());
        return button;
    }

    private Component createCloseDialogButton() {
        JButton button = new JButton("Schließen");
        button.addActionListener(e -> storeSearchParametersAndCloseDialog());
        return button;
    }

    /** Der Anfang des Namens für abgespeicherte Listen aus der erweiterten Suche. */
    public static final String COMPLEX_SEARCH_NAME_START = "Erweiterte Suche";

    private void saveList() {
        List<Vocable> foundVocables = vocableFilter.createFilteredVocablesList();
        if (foundVocables.isEmpty()) {
            informNoVocablesMatchSearchCriteria();
        }
        else {
            gui.saveAsList(foundVocables, COMPLEX_SEARCH_NAME_START);
        }
    }

    private void updateStatistic(List<Vocable> filteredVocablesList) {
        statisticLabel
                .setText("Anzahl Vokabeln: " + NumberString.taupu(filteredVocablesList.size()));
    }

    private void storeSearchParametersAndCloseDialog() {
        vocableFilter.storeInputs(inputs);
        closeDialog();
    }

}
