package de.duehl.vocabulary.japanese.ui.dialog;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.NumberString;
import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.components.MultipleElementsPanel;
import de.duehl.swing.ui.dialogs.base.NonModalFrameDialogBase;
import de.duehl.vocabulary.japanese.common.data.VocablesShuffleType;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.tools.VocableListShuffler;
import de.duehl.vocabulary.japanese.ui.components.VocableViewer;
import de.duehl.vocabulary.japanese.ui.components.button.OtherViewButtonPanel;
import de.duehl.vocabulary.japanese.ui.components.button.ViewButtonPressedReactor;
import de.duehl.vocabulary.japanese.ui.data.MessageSetter;
import de.duehl.vocabulary.japanese.ui.dialog.detail.VocableWithInternaViewer;
import de.duehl.vocabulary.japanese.ui.tools.VocabularyTrainerUiTools;

/**
 * Diese Klasse zeigt eine Menge von Vokabeln in Form eines Vokabelblattes an.
 *
 * @version 1.01     2025-09-22
 * @author Christian Dühl
 */

public class VocabularySheetDialog extends NonModalFrameDialogBase
        implements ViewButtonPressedReactor {

    private static final Dimension DIALOG_DIMENSION = new Dimension(1500, 900);

    private static final int DEFAULT_NUMBER_OF_VOCABLES_IN_ONE_ROW = 3;

    private static final List<Integer> POSSIBLE_COLUMNS =
            CollectionsHelper.buildListFrom(1, 2, 3, 4);

    private static final int MAXIMAL_NUMBER_OF_VOCABLES_FOR_COLUMNS_CHANGE = 100;


    /** Die Programmoptionen. */
    private final Options options;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalDataRequester requester;

    /** Die Liste mit den anzuzeigenden Vokabeln. */
    private final List<Vocable> vocables;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalKanjiDataRequester kanjiRequester;

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /** Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann. */
    private final MessageSetter messageSetter;

    /**
     * Zeigt drei Buttons an, mit denen man eine Menge von Vokabeln in Form einer
     * Einzeldarstellung, in Form einer Liste und in Form eines Vokabelblattes anzeigen lassen
     * kann.
     */
    private final OtherViewButtonPanel otherViewPanel;

    /** Die ScrollPane um die Vokabeln. */
    private JScrollPane vocablesScrollPane;

    /** Der Panel mit der ScrollPane um die Vokabeln. */
    private final JPanel vocablesJPanel;

    /** Die Anzahl der Spalten für die Darstellung als Vokabelblatt. */
    private int numberOfColumns;

    /** Die Auswahlmöglichkeit der Anzahl an Spalten mit den auf einer Seite angezeigten Daten. */
    private final JComboBox<Integer> numberOfColumnsComboBox;

    /** Der Button mit dem man das Einfärben des Hintergrunds ein- und ausschaltet. */
    private final JButton backgroundChangingButton;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param vocables
     *            Die Liste mit den abzufragenden Vokabeln.
     * @param description
     *            Die Beschreibung der Menge von Vokabeln.
     * @param kanjiRequester
     *            Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param informer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     * @param ownLists
     *            Die Verwaltung der eigenen Vokabellisten.
     * @param messageSetter
     *            Das Objekt, welches in der Statusbar der Gui eine Nachricht anzeigen kann.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Anzuzeigendes ProgrammIcon.
     */
    public VocabularySheetDialog(Options options, InternalDataRequester requester,
            List<Vocable> vocables, String description, InternalKanjiDataRequester kanjiRequester,
            LongTimeProcessInformer informer, OwnLists ownLists, MessageSetter messageSetter,
            Point parentLocation, Image programImage) {
        super(parentLocation, programImage,
                createDialogTitle(options, requester, vocables, description), DIALOG_DIMENSION);
        this.options = options;
        this.requester = requester;
        this.kanjiRequester = kanjiRequester;
        this.ownLists = ownLists;
        this.messageSetter = messageSetter;

        vocablesJPanel = new JPanel();
        if (options.isUseVocablesShuffleTypeForShowingListsAndSheetsToo()) {
            VocablesShuffleType type = options.getVocablesShuffleType();
            VocableListShuffler shuffler = new VocableListShuffler(options, vocables, type,
                    requester);
            shuffler.shuffle();
            this.vocables = shuffler.getVocables();
        }
        else {
            this.vocables = vocables;
        }

        otherViewPanel = new OtherViewButtonPanel(vocables, (ViewButtonPressedReactor) this,
                description, requester, options, kanjiRequester, informer, ownLists, messageSetter,
                parentLocation, programImage);
        otherViewPanel.disableVokabelblattDarstellungButton();

        numberOfColumns = options.getNumberOfColumnsOfBlattDarstellung();
        if (numberOfColumns < 1) {
            numberOfColumns = DEFAULT_NUMBER_OF_VOCABLES_IN_ONE_ROW;
        }
        numberOfColumnsComboBox = new JComboBox<>();

        backgroundChangingButton = new JButton();

        init();
        rearrangeVocablePart();
        fillDialog();
    }

    private static String createDialogTitle(Options options, InternalDataRequester requester,
            List<Vocable> vocables, String description) {
        String frontTitle = "Anzeige mit den " + NumberString.taupu(vocables.size())
                + " Vokabeln aus " + description;
        return VocabularyTrainerUiTools.generateTitleWithVocabularyTestSuccesss(options, requester,
                vocables, frontTitle);
    }

    private void init() {
        initVocablesPanel();
        initNumberOfColumnsComboBox();
        initBackgroundChangingButton();
    }

    private void initVocablesPanel() {
        vocablesJPanel.setLayout(new BorderLayout());
    }

    private void initNumberOfColumnsComboBox() {
        numberOfColumnsComboBox.setAlignmentX(JLabel.CENTER);
        numberOfColumnsComboBox.setAlignmentY(JLabel.CENTER);
        ((JLabel) numberOfColumnsComboBox.getRenderer()).setHorizontalAlignment(
                SwingConstants.RIGHT);

        List<Integer> possibleColumns = new ArrayList<>();
        possibleColumns.addAll(POSSIBLE_COLUMNS);
        if (!POSSIBLE_COLUMNS.contains(numberOfColumns)) {
            possibleColumns.add(numberOfColumns);
        }
        for (int value : possibleColumns) {
            numberOfColumnsComboBox.addItem(value);
        }

        numberOfColumnsComboBox.setSelectedItem(numberOfColumns);
        numberOfColumnsComboBox.addActionListener(e -> rearrangePageAfterChangedNumbers());
    }

    private void initBackgroundChangingButton() {
        setBackgroundChangingButtonTitle();
        backgroundChangingButton.addActionListener(e -> toggleBackground());
    }

    private void setBackgroundChangingButtonTitle() {
        String buttonText = ""
                + "Hintergrund "
                + (options.isColorVocableDependingOnLastSuccessWhenViewingVocable()
                        ? "aus"
                        : "ein")
                + "schalten";
        backgroundChangingButton.setText(buttonText);
    }

    private void toggleBackground() {
        boolean onOrOff = options.isColorVocableDependingOnLastSuccessWhenViewingVocable();
        onOrOff = !onOrOff;
        options.setColorVocableDependingOnLastSuccessWhenViewingVocable(onOrOff);
        setBackgroundChangingButtonTitle();
        rearrangePageAfterChangedNumbers() ;
    }

    private void rearrangePageAfterChangedNumbers() {
        numberOfColumns = (Integer) numberOfColumnsComboBox.getSelectedItem();
        rearrangeVocablePart();
        options.setNumberOfColumnsOfBlattDarstellung(numberOfColumns);
    }

    private void rearrangeVocablePart() {
        List<Component> vocablePanes = createVocablePanes();
        JPanel panel = new MultipleElementsPanel<Component>(vocablePanes, numberOfColumns, 1);
        vocablesScrollPane = GuiTools.createScrollPane(panel);

        vocablesJPanel.removeAll();
        vocablesJPanel.add(vocablesScrollPane, BorderLayout.CENTER);

        //vocablesScrollPane.repaint();
        vocablesJPanel.repaint();
        vocablesJPanel.validate();
        vocablesJPanel.invalidate();

        GuiTools.scrollScrollbarToMinimumLater(vocablesScrollPane);
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createUpperPart(),  BorderLayout.NORTH);
        add(vocablesJPanel, BorderLayout.CENTER);
        add(createButtonsPart(),  BorderLayout.SOUTH);
    }

    private Component createUpperPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        panel.add(backgroundChangingButton, BorderLayout.WEST);
        if (vocables.size() <= MAXIMAL_NUMBER_OF_VOCABLES_FOR_COLUMNS_CHANGE) {
            panel.add(createNumberOfColumnsComboBoxWithTitelPart(), BorderLayout.EAST);
        }

        return panel;
    }

    private Component createNumberOfColumnsComboBoxWithTitelPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(new JLabel("Anzahl Spalten "), BorderLayout.WEST);
        panel.add(numberOfColumnsComboBox, BorderLayout.CENTER);

        return panel;
    }

    private List<Component> createVocablePanes() {
        List<Component> vocablePanes = new ArrayList<>();

        for (Vocable vocable : vocables) {
            Component vocablePane = createVocablePane(vocable);
            vocablePanes.add(vocablePane);
        }

        return vocablePanes;
    }

    private Component createVocablePane(Vocable vocable) {
        VocableViewer viewer = new VocableViewer(options);
        viewer.showVocable(vocable, requester.getInternalDataForVocable(vocable));
        Component viewerPanel = viewer.getPanel();
        if (options.isShowDetailsButtonInVokabelblattDarstellung()) {
            JPanel panel = new JPanel();
            panel.setLayout(new BorderLayout());
            panel.add(viewerPanel, BorderLayout.CENTER);
            panel.add(createDetailsButton(vocable), BorderLayout.SOUTH);
            return panel;
        }
        else {
            return viewerPanel;
        }
    }

    private Component createDetailsButton(Vocable vocable) {
        JButton button = new JButton("Details anzeigen");
        button.addActionListener(e -> showDetails(vocable));
        return button;
    }

    private void showDetails(Vocable vocable) {
        VocableWithInternaViewer viewer = new VocableWithInternaViewer(options, requester, vocable,
                ownLists, kanjiRequester, messageSetter, getLocation(), getProgramImage());
        viewer.setVisible(true);
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.EAST);
        panel.add(GuiTools.centerHorizontal(otherViewPanel.getPanel()), BorderLayout.CENTER);

        return panel;
    }

    private Component createQuitButton() {
        JButton button = new JButton("Beenden");
        button.addActionListener(e -> quit());
        return button;
    }

    private void quit() {
        closeDialog();
    }

    /** Wird aufgerufen, wenn die Vokabeln in einer anderen Ansicht angezeigt werden. */
    @Override
    public void otherViewOpened() {
        closeDialog();
    }

}
