package de.duehl.vocabulary.japanese.ui.dialog.detail.findlists;

import java.awt.Component;
import java.awt.Point;
import java.util.List;

import de.duehl.basics.text.Text;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.text.TextViewer;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;

/**
 * Diese Klasse findet zu einer Vokabel alle eigenen Listen, in denen die Vokabel vorkommt.
 *
 * @version 1.01     2025-07-14
 * @author Christian Dühl
 */

class ListsWithVocableFinder {

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /**
     * Der modale oder nicht modale Dialog bzw., Frame, der den Panel nutzt, welcher diese Klasse
     * verwendet.
     */
    private final Component thisPanelUsingDialog;

    /**
     * Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese Klasse
     * verwendet.
     */
    private final Point location;

    /**
     * Konstruktor.
     *
     * @param ownLists
     *            Die Verwaltung der eigenen Vokabellisten.
     * @param thisPanelUsingDialog
     *            Der modale oder nicht modale Dialog bzw., Frame, der den Panel nutzt, welcher
     *            diese Klasse verwendet.
     * @param location
     *            Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese
     *            Klasse verwendet.
     */
    public ListsWithVocableFinder(OwnLists ownLists,
            Component thisPanelUsingDialog, Point location) {
        this.ownLists = ownLists;
        this.thisPanelUsingDialog = thisPanelUsingDialog;
        this.location = location;
    }

    /** Zeigt zur übergebenen Vokabel alle Listen an, die diese Vokabel enthalten. */
    public void findListsWithVocable(Vocable vocable) {
        String kanji = vocable.getKanji();
        String kana = vocable.getKana();
        String vocableText = kanji.isBlank()
                ? kana
                : kanji + " (" + kana + ")";

        List<String> names =  ownLists.findNamesOfListsWithVocable(vocable);
        if (names.isEmpty()) {
            GuiTools.informUser(thisPanelUsingDialog, "Keine Liste gefunden.", ""
                    + vocableText + "\n\n"
                    + "Die Vokabel kommt in keiner Liste vor.");
        }
        else {
            String text = "Die Vokabel " + vocableText + " kommt in den folgenden Listen vor:\n\n"
                    + Text.joinWithLineBreak(names);
            TextViewer viewer = new TextViewer("Listen mit der Vokabel", location);
            viewer.setText(text);
            viewer.setVisible(true);
        }
    }

}
