package de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjitest.components;

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.JPanel;

import de.duehl.basics.text.Text;
import de.duehl.swing.ui.components.selections.SelectionsHelper;
import de.duehl.swing.ui.components.selections.StringSelection;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalKanjiData;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;

/**
 * Diese Klasse stellt ein einzelnes Kanji Panel dar.
 *
 * Gezeigt werden Das Zeichen selbst, die deutsche Bedeutung, die ON-Lesungen und die kun-Lesungen.
 *
 * @version 1.01     2024-12-13
 * @author Christian Dühl
 */

public class KanjiPanel {

    /** Das Label auf dem das Kanji dargestellt wird. */
    private final KanjiLabel kanjiLabel;

    /** Gibt an, ob auch die internen Daten zum Kanji angezeigt werden sollen. */
    private boolean showInternalKanjiData;

    /**
     * Gibt an, ob das Zeichen auch nochmal in einem Viewer angezeigt werden soll, damit es
     * heraus kopiert werden kann.
     */
    private boolean showKanjiAsViewerToo;

    /** Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen kann. */
    private InternalKanjiDataRequester requester;

    /** Der Panel auf dem das Kanji dargestellt wird. */
    private final JPanel panel;

    private final StringSelection kanjiAsCharacterSeletion;
    private final StringSelection germanMeaningSelection;
    private final StringSelection onLesungenSelection;
    private final StringSelection kunLesungenSelection;

    private final StringSelection numberInKanjiAndKanaBookSelection;
    private final StringSelection pageNumberInKanjiAndKanaBookSelection;

    private final StringSelection firstSeenDateSelection;
    private final StringSelection testCountSelection;
    private final StringSelection correctTestCountSelection;
    private final StringSelection lastTestDatetSelection;
    private final StringSelection lastCorrectTestDateSelection;
    private final StringSelection lastTenTestResultsSelection;

    /** Konstruktor. */
    public KanjiPanel() {
        kanjiLabel = new KanjiLabel();
        kanjiLabel.setBiggerFontSize(200); // Default: 400

        showInternalKanjiData = false;
        showKanjiAsViewerToo = false;


        panel = new JPanel();

        kanjiAsCharacterSeletion = new StringSelection("Das Kanji (noch einmal zum rauskopieren)");
        germanMeaningSelection = new StringSelection("Deutsche Bedeutung");
        onLesungenSelection = new StringSelection("ON-Lesungen");
        kunLesungenSelection = new StringSelection("kun-Lesungen");

        numberInKanjiAndKanaBookSelection = new StringSelection(
                "Nummer im im Buch 'Kanji und Kana' von Wolfgang Hadamitzky");
        pageNumberInKanjiAndKanaBookSelection = new StringSelection(
                "Seite im im Buch 'Kanji und Kana' von Wolfgang Hadamitzky");

        firstSeenDateSelection = new StringSelection("Zuerst gesehen");
        testCountSelection = new StringSelection("Anzahl Test");
        correctTestCountSelection = new StringSelection("Anzahl erfolgreicher Test");
        lastTestDatetSelection = new StringSelection("zuletzt getestet am");
        lastCorrectTestDateSelection = new StringSelection("zuletzt erfolgreich getestet am");
        lastTenTestResultsSelection = new StringSelection("Erfolg der letzten Tests");

        init();
    }

    private void init() {
        initAssetFocusableViewer(kanjiAsCharacterSeletion);
        initAssetFocusableViewer(kanjiAsCharacterSeletion);
        initAssetFocusableViewer(germanMeaningSelection);
        initAssetFocusableViewer(onLesungenSelection);
        initAssetFocusableViewer(kunLesungenSelection);

        initAssetFocusableViewer(numberInKanjiAndKanaBookSelection);
        initAssetFocusableViewer(pageNumberInKanjiAndKanaBookSelection);

        initAssetFocusableViewer(firstSeenDateSelection);
        initAssetFocusableViewer(testCountSelection);
        initAssetFocusableViewer(correctTestCountSelection);
        initAssetFocusableViewer(lastTestDatetSelection);
        initAssetFocusableViewer(lastCorrectTestDateSelection);
        initAssetFocusableViewer(lastTenTestResultsSelection);
    }

    private void initAssetFocusableViewer(StringSelection selection) {
        SelectionsHelper.initSelectionAsViewer(selection);
        selection.setFocusable(true);
    }

    public void showKanjiAsViewerToo() {
        showKanjiAsViewerToo = true;
    }

    /**
     * Legt fest, dass auch die internen Daten zum Kanji angezeigt werden sollen.
     *
     * @param requester
     *            Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     *            kann.
     */
    public void showInternalKanjiDataToo(InternalKanjiDataRequester requester) {
        showInternalKanjiData = true;
        this.requester = requester;
    }

    /**
     * Erzeugt die Gui.
     *
     * Möchte man die internen Daten anzeigen, so muss man zuvor die Methode
     * showInternalKanjiDataToo(requester) aufrufen!
     */
    public void createGui() {
        panel.setLayout(new BorderLayout());
        //GuiTools.createTitle(panel); Das sollte man wohl außerhalb entscheiden.

        panel.add(kanjiLabel.getLabel(), BorderLayout.CENTER);
        panel.add(createGermanMeaningOnLesungenAndKunLesungenAndPerhapsInternalDataPart(),
                BorderLayout.SOUTH);
    }

    private Component createGermanMeaningOnLesungenAndKunLesungenAndPerhapsInternalDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        if (showKanjiAsViewerToo) {
            panel.add(kanjiAsCharacterSeletion.getPanel());
        }
        panel.add(createGermanMeaningOnLesungenAndKunLesungenPart());
        if (showInternalKanjiData) {
            panel.add(createInternalKanjiDataPart());
        }
        return panel;
    }

    private Component createGermanMeaningOnLesungenAndKunLesungenPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(germanMeaningSelection.getPanel());
        panel.add(onLesungenSelection.getPanel());
        panel.add(kunLesungenSelection.getPanel());

        return panel;
    }

    private Component createInternalKanjiDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(numberInKanjiAndKanaBookSelection.getPanel());
        panel.add(pageNumberInKanjiAndKanaBookSelection.getPanel());

        panel.add(firstSeenDateSelection.getPanel());
        panel.add(testCountSelection.getPanel());
        panel.add(correctTestCountSelection.getPanel());
        panel.add(lastTestDatetSelection.getPanel());
        panel.add(lastCorrectTestDateSelection.getPanel());
        panel.add(lastTenTestResultsSelection.getPanel());

        return panel;
    }

    /** Getter für den Panel auf dem das Kanji dargestellt wird. */
    public JPanel getPanel() {
        return panel;
    }

    /** Zeigt das Kanji an. Kann mehrfach aufgerufen werden. */
    public void showKanji(Kanji kanji) {
        kanjiLabel.showKanji(kanji);

        kanjiAsCharacterSeletion.setText(kanji.getCharacter());
        germanMeaningSelection.setText(kanji.getGermanMeaning());
        onLesungenSelection.setText(Text.joinWithCommaAndBlank(kanji.getOnLesungen()));
        kunLesungenSelection.setText(Text.joinWithCommaAndBlank(kanji.getKunLesungen()));

        if (showInternalKanjiData) {
            numberInKanjiAndKanaBookSelection.setText(kanji.getNumberInKanjiAndKanaBook());
            pageNumberInKanjiAndKanaBookSelection.setText(kanji.getPageNumberInKanjiAndKanaBook());

            InternalAdditionalKanjiData data = requester.getInternalDataForKanji(kanji);

            firstSeenDateSelection.setText(data.getFirstSeenDate().toString());
            testCountSelection.setText(data.getTestCount());
            correctTestCountSelection.setText(data.getCorrectTestCount());
            lastTestDatetSelection.setText(data.getLastTestDate().toString());
            lastCorrectTestDateSelection.setText(data.getLastCorrectTestDate().toString());
            lastTenTestResultsSelection.setText(data.getLastTenTestResultsAsStorageString());
        }
    }

}
