package de.duehl.vocabulary.japanese.ui.dialog.table.katakana;

import java.awt.Color;
import java.awt.Component;

import javax.swing.JLabel;
import javax.swing.JTable;

import de.duehl.swing.ui.tables.DifferentBackgroundsTableRendererWithChangingSize;

/**
 * Diese Klasse ist für das Rendern (die optische Darstellung) der Tabellenzellen zuständig. Sie
 * wurde von DifferentBackgroundsTableRendererWithChangingSize abgeleitet, um nicht für jede Zelle
 * eine neue Klasse zu erzeugen.
 *
 * Man kann den Vordergrund Hintergrund der Darstellung abhängig vom Abfrageerfolg der letzten
 * Abfragen gestalten.
 *
 * @version 1.01     2025-02-02
 * @author Christian Dühl
 */

public class KatakanaTableRenderer extends DifferentBackgroundsTableRendererWithChangingSize {

    private static final long serialVersionUID = 1L;

    /**
     * Gibt an, ob der Hintergrund einer Zeile vom Abfrageerfolg der letzten Abfragen abhängig sein
     * soll.
     */
    private boolean colorForegroundDependingOnLastSuccess;

    /**
     * Gibt an, ob der Vordergrund einer Zeile vom Abfrageerfolg der letzten Abfragen abhängig sein
     * soll.
     */
    private boolean colorBackgroundDependingOnLastSuccess;

    /**
     * Der Konstruktor, hier werden unsere Maps erzeugt.
     *
     * @param foregroundColor
     *            Vordergrundfarbe
     * @param backgroundColor
     *            Hintergrundfarbe
     */
    public KatakanaTableRenderer(Color foregroundColor, Color backgroundColor) {
        super(foregroundColor, backgroundColor);
        setUseDifferentBackgrounds(true);
    }

    /**
     * Schaltet um, ob der Hintergrund einer Zeile vom Abfrageerfolg der letzten Abfragen abhängig
     * sein soll.
     */
    public void toggleColorForegroundDependingOnLastSuccess() {
        colorForegroundDependingOnLastSuccess = !colorForegroundDependingOnLastSuccess;
        setUseDifferentBackgrounds();
    }

    /**
     * Schaltet um, ob der Vordergrund einer Zeile vom Abfrageerfolg der letzten Abfragen abhängig
     * sein soll.
     */
    public void toggleColorBackgroundDependingOnLastSuccess() {
        colorBackgroundDependingOnLastSuccess = !colorBackgroundDependingOnLastSuccess;
        setUseDifferentBackgrounds();
    }

    private void setUseDifferentBackgrounds() {
        setUseDifferentBackgrounds(
                !colorForegroundDependingOnLastSuccess && !colorBackgroundDependingOnLastSuccess);
    }

    /**
     * Liefert eine Komponente für die anzuzeigende Zelle zurück. Die Hauptarbeit lassen wir
     * super.getTableCellRendererComponent(...) machen und hängen nur ein wenig Code dahinter.
     *
     * @param table
     *            Die Tabelle
     * @param value
     *            Das anzuzeigende Objekt (im Normalfall ein String)
     * @param isSelected
     *            Zeigt an, ob das Objekt selektiert wurde.
     * @param hasFocus
     *            Zeigt an, ob der Focus auf dieser Zelle liegt.
     * @param row
     *            Zeile in der die Zelle liegt.
     * @param column
     *            Spalte in der die Zelle liegt.
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
            boolean hasFocus, int row, int column) {
        int perhapsSortedRow;
        if (colorBackgroundDependingOnLastSuccess || colorForegroundDependingOnLastSuccess) {
            perhapsSortedRow = table.convertRowIndexToModel(row);
            /*
             * vgl. https://stackoverflow.com/questions/10321038/
             * jtable-clickable-column-sorting-sorting-sorts-content-of-cells-but-doesnt-upd
             *
             * Siehe auch
             * https://www.codejava.net/java-se/swing/6-techniques-for-sorting-jtable-you-should-know
             */
        }
        else {
            perhapsSortedRow = row;
        }

        return getTableCellRendererComponentInternal(table, value, isSelected, hasFocus,
                perhapsSortedRow, column);
    }

    private Component getTableCellRendererComponentInternal(JTable table, Object value,
            boolean isSelected, boolean hasFocus, int row, int column) {
        JLabel createdLabel = (JLabel) super.getTableCellRendererComponent(table, value, isSelected,
                hasFocus, row, column);

        /*
         * Wenn die Zeile selected ist und die Farben dabei vertauscht werden sollen, dann sind sie
         * im erzeugten Label createdLabel bereits vertauscht. Daher muss für diese Fälle im
         * Folgenden manchmal zurück getauscht werden.
         */

        KatakanaTableModel model = (KatakanaTableModel) table.getModel();

        Color foreground;
        Color background;

        if (colorForegroundDependingOnLastSuccess && colorBackgroundDependingOnLastSuccess) {
            foreground = model.determineRowForegroundColor(row);
            background = model.determineRowBackgroundColor(row);
        }
        else if (colorForegroundDependingOnLastSuccess) {
            foreground = model.determineRowForegroundColor(row);
            if (isSelected && isUseSwitchedColorsForSelection()) {
                background = createdLabel.getForeground();
            }
            else {
                background = createdLabel.getBackground();
            }
        }
        else if (colorBackgroundDependingOnLastSuccess) {
            //foreground = createdLabel.getForeground();
            foreground = Color.BLACK;
            background = model.determineRowBackgroundColor(row);
        }
        else {
            if (isSelected && isUseSwitchedColorsForSelection()) {
                foreground = createdLabel.getBackground();
                background = createdLabel.getForeground();
            }
            else {
                foreground = createdLabel.getForeground();
                background = createdLabel.getBackground();
            }
        }

        if (isSelected && isUseSwitchedColorsForSelection()) {
            createdLabel.setForeground(background);
            createdLabel.setBackground(foreground);
        }
        else {
            createdLabel.setForeground(foreground);
            createdLabel.setBackground(background);
        }

        return createdLabel;
    }

}
