package de.duehl.vocabulary.japanese.ui.tools;

import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;

import de.duehl.basics.text.NumberString;
import de.duehl.swing.ui.resources.IconLoader;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.common.ui.resources.IconDefinitions;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.data.symbol.Hiragana;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.data.symbol.Katakana;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;
import de.duehl.vocabulary.japanese.logic.success.HiraganaListTestSuccesssCalculator;
import de.duehl.vocabulary.japanese.logic.success.KanjiListTestSuccesssCalculator;
import de.duehl.vocabulary.japanese.logic.success.KatakanaListTestSuccesssCalculator;
import de.duehl.vocabulary.japanese.logic.success.VocabularyTestSuccesssCalculator;
import de.duehl.vocabulary.japanese.logic.symbol.kana.internal.InternalKanaDataRequester;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;

/**
 * Diese Klasse stellt Hilfsmethoden für die Gui zur Verfügung.
 *
 * @version 1.01     2025-02-02
 * @author Christian Dühl
 */

public class VocabularyTrainerUiTools {

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Vokabeln erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param vocables
     *            Die Liste mit den abzufragenden Vokabeln.
     * @param frontTitle
     *            Der vordere Teil des Titels.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Vokabeln in
     *         Klammern dahinter.
     */
    public static String generateTitleWithVocabularyTestSuccesss(Options options,
            InternalDataRequester requester, List<Vocable> vocables, String frontTitle) {
        VocabularyTestSuccesssCalculator calculator =
                new VocabularyTestSuccesssCalculator(vocables, options, requester);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return frontTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Kanji erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param vocables
     *            Die Liste mit den abzufragenden Kanji.
     * @param frontTitle
     *            Der vordere Teil des Titels.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Kanji in
     *         Klammern dahinter.
     */
    public static String generateTitleWithKanjiTestSuccesss(Options options,
            InternalKanjiDataRequester requester, List<Kanji> kanjiToTest, String frontTitle) {
        KanjiListTestSuccesssCalculator calculator = new KanjiListTestSuccesssCalculator(
                kanjiToTest, options, requester);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return frontTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Hiragana erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einem Hiragana die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param hiraganaToTest
     *            Die Liste mit den abzufragenden Hiragana.
     * @param testTitle
     *            Der Titels.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Hiragana in
     *         Klammern dahinter.
     */
    public static String generateTitleWithHiraganaTestSuccesss(Options options,
            InternalKanaDataRequester requester, List<Hiragana> hiraganaToTest, String testTitle) {
        HiraganaListTestSuccesssCalculator calculator = new HiraganaListTestSuccesssCalculator(
                hiraganaToTest, options, requester);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return testTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt den Titel für einen Dialog, welcher auch den Erfolg in Prozent
     * anzeigt, der bei den letzten Abfragen der Katakana erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einem Katakana die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param hiraganaToTest
     *            Die Liste mit den abzufragenden Katakana.
     * @param testTitle
     *            Der Titels.
     * @return Der Titel mit Prozentangabe des Erfolgs bei den letzten Abfragen der Katakana in
     *         Klammern dahinter.
     */
    public static String generateTitleWithKatakanaTestSuccesss(Options options,
            InternalKanaDataRequester requester, List<Katakana> katakanaToTest, String testTitle) {
        KatakanaListTestSuccesssCalculator calculator = new KatakanaListTestSuccesssCalculator(
                katakanaToTest, options, requester);
        calculator.calculate();
        String percent = calculator.getPercentTextWithTwoDigitsAfterComma();

        return testTitle + " (" + percent + "%)";
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Vokabeln
     * erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param vocables
     *            Die Liste mit den abzufragenden Vokabeln.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Vokabeln.
     */
    public static double createLastTenTestsPercent(Options options, InternalDataRequester requester,
            List<Vocable> vocables) {
        VocabularyTestSuccesssCalculator calculator =
                new VocabularyTestSuccesssCalculator(vocables, options, requester);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Kanji
     * erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param kanjiList
     *            Die Liste mit den abzufragenden Kanji.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Kanji.
     */
    public static double createLastTenKanjiTestsPercent(Options options,
            InternalKanjiDataRequester requester, List<Kanji> kanjiList) {
        KanjiListTestSuccesssCalculator calculator = new KanjiListTestSuccesssCalculator(kanjiList,
                options, requester);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Hiragana
     * erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einem Hiragana die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param hiraganaList
     *            Die Liste mit den abzufragenden Hiragana.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Hiragana.
     */
    public static double createLastTenHiraganaTestsPercent(Options options,
            InternalKanaDataRequester requester, List<Hiragana> hiraganaList) {
        HiraganaListTestSuccesssCalculator calculator = new HiraganaListTestSuccesssCalculator(
                hiraganaList, options, requester);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Diese Methode erzeugt die Prozentzahl des Erfolgs, der bei den letzten Abfragen der Katakana
     * erzielt wurde.
     *
     * @param options
     *            Die Programmoptionen.
     * @param requester
     *            Das Objekt das zu einem Katakana die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param katakanaList
     *            Die Liste mit den abzufragenden Katakana.
     * @return Die Prozentzahl des Erfolgs bei den letzten Abfragen der Katakana.
     */
    public static double createLastTenKatakanaTestsPercent(Options options,
            InternalKanaDataRequester requester, List<Katakana> katakanaList) {
        KatakanaListTestSuccesssCalculator calculator = new KatakanaListTestSuccesssCalculator(
                katakanaList, options, requester);
        calculator.calculate();
        return calculator.getPercent();
    }

    /**
     * Erzeugt einen Button mit einem Icon.
     *
     * @param pictureIdentifier
     *            Bezeichner des Bildes, im Normalfall eine der öffentlichen Konstanten aus
     *            IconDefinitions.
     * @param listener
     *            Wird beim Betätigen des Buttons ausgeführt.
     * @param toolTipText
     *            Wird angezeigt, wenn die Maus eine Weile über dem Button verweilt.
     * @return Der erzeugte Button.
     */
    public static JButton createPicturedButton(String pictureIdentifier, ActionListener listener,
            String toolTipText) {
        JButton button = new JButton(loadIcon(pictureIdentifier));
        button.setBorder(BorderFactory.createRaisedBevelBorder());
        button.addActionListener(listener);
        button.setToolTipText(toolTipText);
        return button;
    }

    private static Icon loadIcon(String imageIdentifier) {
        IconDefinitions iconDefinitions = new IconDefinitions();
        IconLoader iconLoader = iconDefinitions.createIconLoader();
        return iconLoader.loadIcon(imageIdentifier);
    }

    /**
     * Erzeugt für Abschlussdialoge von Abfragen die richtige Beschreibung.
     *
     * TODO: Später soll auch eine passende Fumiko-Chan mitgeliefert werden, dann muss ich eine
     * kleine Datenklasse mit der Nachricht und dem Bild programmieren und zurückgeben.
     *
     * @param whatWasTestet
     *            Was getestet wurde,. z.B. "Vokabeln" oder "Kanji".
     * @param testVerb
     *            Wie man das beantworten des Tests bezeichnet, z.B. "übersetzt" oder
     *            "beantwortet".
     * @param total
     *            Die Anzahl aller abgefragten Dinge.
     * @param correct
     *            Die Anzahl der richtig abgefragten Dinge.
     * @return Die Beschreibung der Leistung des Benutzers.
     */
    public static String createAllTestsMessage(String whatWasTestet, String testVerb, int total,
            int correct) {
        String correctFromTotal = "Sie haben " + NumberString.taupu(correct) + " von "
                + NumberString.taupu(total) + " " + whatWasTestet + " richtig " + testVerb + ".";

        String successDescription;
        if (total == correct) {
            successDescription = "Sehr gut, alles richtig " + testVerb + "!";
        }
        else {
            double percent = NumberString.percentAsNumber(correct, total);
            if (percent >= 90) {
                successDescription = "Das war schon sehr gut!";
            }
            else if (percent >= 75) {
                successDescription = "Ein gutes Ergebnis!";
            }
            else if (percent >= 50) {
                successDescription = "Immerhin war mindestens die Hälfte richtig.";
            }
            else if (percent >= 25) {
                successDescription = "Es wird, aber es gibt noch viel Luft nach oben.";
            }
            else {
                successDescription = "Üben, üben, üben ...";
            }
        }

        return correctFromTotal + "<br/>" + successDescription;
    }

}
