package de.duehl.basics.persistence;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.InvalidPropertiesFormatException;
import java.util.Properties;

import de.duehl.basics.io.FileHelper;

/**
 * Diese Klasse regelt den Umgang mit Informationen, die zwischen zwei Läufen des Programms
 * bewahrt werden sollen. Sie stellt Methoden bereit, diese Daten zu speichern und zu lesen.
 *
 * @version 1.01     2017-03-28
 * @author Christian Dühl
 */

public class SessionManagerHelper {

    /** Name der Datei, in der die Informationen gespeichert und aus der gelesen wird. */
    private final String sessionfileName;

    /**
     * Konstruktor.
     *
     * @param sessionfileName
     *            Name der Datei, in der die Informationen gespeichert und aus der gelesen wird.
     */
    public SessionManagerHelper(String sessionfileName) {
        this.sessionfileName = sessionfileName;
    }

    /**
     * Liest das Sessionfile ein und gibt ein passendes Properties Objekt zurück oder null, falls
     * es nicht gefunden wurde oder es beim Einlesen Probleme gab.
     *
     * @return Properties des Sessionfiles oder null, falls das Sessionfile nicht existiert.
     * @throws RuntimeException
     *             Falls das Sessionfile nicht existiert oder ein Fehler auftrat.
     */
    public Properties readProperties() {
        try {
            return tryToReadProperties();
        }
        catch (FileNotFoundException exception) {
            /*
             * Hier keine Fehlermeldung, da es ja nicht schlimm ist, wenn es die Datei noch nicht
             * gibt!
             */
            return null;
        }
        catch (Exception exception) {
            throw new RuntimeException("Fehler beim Einlesen des Sessionfiles " + sessionfileName
                    + ".", exception);
        }
    }

    /**
     * Liest das Sessionfile ein und gibt ein passendes Properties Objekt zurück oder null, falls
     * es nicht gefunden wurde oder es beim Einlesen Probleme gab.
     *
     * @return Properties des Sessionfiles oder null, falls das Sessionfile nicht existiert.
     * @throws IOException
     *             Wenn Beim Einlesen der XML-Datei ein Fehler auftrat.
     * @throws InvalidPropertiesFormatException
     *             Wenn Beim Einlesen der XML-Datei ein Fehler auftrat.
     * @throws FileNotFoundException
     *             Falls das Sessionfile nicht existiert.
     */
    private Properties tryToReadProperties() throws IOException,
            InvalidPropertiesFormatException, FileNotFoundException {
        Properties properties = new Properties();
        properties.loadFromXML(new FileInputStream(sessionfileName));
        return properties;
    }

    /**
     * Speichert die Properties als XML-Datei ab.
     *
     * @param properties
     *            Die zu speichernden Properties.
     * @throws RuntimeException
     *             Wenn beim Einlesen der XML-Datei ein Fehler auftrat oder falls das Sessionfile
     *             nicht existiert.
     */
    public void writeProperties(Properties properties) {
        try {
            tryToWriteProperties(properties);
        }
        catch (FileNotFoundException exception) {
            throw new RuntimeException("Sessionfile " + sessionfileName + " wurde  nicht gefunden: "
                    + exception.getMessage(), exception);
        }
        catch (IOException exception) {
            throw new RuntimeException("Sessionfile " + sessionfileName + " konnte nicht "
                    + "geschrieben werden: " + exception.getMessage(), exception);
        }
    }

    /**
     * Speichert die Properties als XML-Datei ab.
     *
     * @param properties
     *            Die zu speichernden Properties.
     * @throws IOException
     * @throws FileNotFoundException
     */
    private void tryToWriteProperties(Properties properties)
            throws IOException, FileNotFoundException {
        properties.storeToXML(new FileOutputStream(sessionfileName), "");
    }

    /** Prüft, ob die Session-Datei existiert. */
    public boolean sessionfileExists() {
        return FileHelper.isFile(sessionfileName);
    }

}
