package de.duehl.basics.regex;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/*
 * Copyright 2023 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

/**
 * Diese Klasse sucht in einem Text alle Teile, auf welche das Pattern passt und ersetzt diese
 * durch einen Platzhalter.
 *
 * @version 1.01     2023-09-08
 * @author Christian Dühl
 */

public class MultipleTextPartsWithRegexReplacer {

    /** Der Text, welcher hier während der Suche verändert wird. */
    private String ent;

    /** Das Pattern mit dem im Text gesucht wird. */
    private final Pattern pattern;

    /** Der Platzhalter der für alle gefundenen Abschnitte im Text eingefügt wird. */
    private final String placeholder;

    /** Die Liste mit den Fundstücken, welche im Text ersetzt wurden. */
    private List<String> foundParts;

    /**
     * Konstruktor.
     *
     * @param text
     *            Der Text, welcher hier während der Suche verändert wird.
     * @param pattern
     *            Das Pattern mit dem im Text gesucht wird.
     * @param placeholder
     *            Der Platzhalter der für alle gefundenen Abschnitte im Text eingefügt wird.
     */
    public MultipleTextPartsWithRegexReplacer(String text, Pattern pattern, String placeholder) {
        this.ent = text;
        this.pattern = pattern;
        this.placeholder = placeholder;
    }

    /** Führt das Ersetzen durch. */
    public void replace() {
        foundParts = new ArrayList<>();
        Matcher matcher = pattern.matcher(ent);
        while (matcher.find()) {
            String front = ent.substring(0, matcher.start());
            String rear = ent.substring(matcher.end());
            ent = front + placeholder + rear;

            String matchedPhrase = matcher.group();
            foundParts.add(matchedPhrase);

            matcher = pattern.matcher(ent);
        }
    }

    /** Getter für den Text, welcher hier während der Suche verändert wurde. */
    public String getEnt() {
        return ent;
    }

    /** Getter für die Liste mit den Fundstücken, welche im Text ersetzt wurden. */
    public List<String> getFoundParts() {
        return foundParts;
    }

}
