package de.duehl.basics.text.extract.tools;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse zerlegt für den Prozess, welcher brauchbare Worte aus einem Text extrahiert, den
 * Text in Zeilen und säubert in diesen Zeilen vorbereitend bestimmte Dinge.
 *
 * @version 1.01     2019-04-25
 * @author Christian Dühl
 */

public class PrepareLines {

    private static final List<String> NON_WORD_CHARS = CollectionsHelper.buildListFrom("(", ")",
            "[", "]", "{", "}", "<", ">", "„", "“",
            "‘", "'", "´", "`",
            "\"", "/", "|", ",", ".", ";", ":", "!", "?", "§", "$", "%", "&", "=",
            "_", "#", "²", "³", "^", "°", "*", "+", "~"
            // "@" nicht, Worte mit Klammeraffen werden ganz ausortiert.
            );

    /** Der Text aus dem die Zeilen extrahiert werden sollen. */
    private final String text;

    /** Die Zeilen des Textes, ein Zwischenprodukt, welches hier erstellt wird. */
    private List<String> lines;

    /**
     * Konstruktor.
     *
     * @param text
     *            Der Text aus dem die Zeilen extrahiert werden sollen.
     */
    public PrepareLines(String text) {
        this.text = text;

        splitTextIntoLines();
        removeBbCodeAndNonWordCharactersFromLines();
    }

    private void splitTextIntoLines() {
        lines = Text.splitByLineBreaks(text);
    }

    private void removeBbCodeAndNonWordCharactersFromLines() {
        lines = removeBbCodeAndNonWordCharactersFromLinesInternal();
    }

    private List<String> removeBbCodeAndNonWordCharactersFromLinesInternal() {
        List<String> cleanedLines = new ArrayList<>();

        for (String line : lines) {
            cleanedLines.add(removeBbCodeAndNonWordCharacters(line));
        }

        return cleanedLines;
    }

    private String removeBbCodeAndNonWordCharacters(String line) {
        String cleanedLine = line;
        cleanedLine = removeBbCode(cleanedLine);
        cleanedLine = removeNonWordCharacters(cleanedLine);
        return cleanedLine;
    }

    private String removeBbCode(String line) {
        String cleanedLine = line;

        cleanedLine = cleanedLine.replaceAll("\\[/?[^\\]]+\\]", " ");

        return cleanedLine;
    }

    private String removeNonWordCharacters(String line) {
        String cleanedLine = line;

        for (String nonWordCharacter : NON_WORD_CHARS) {
            cleanedLine = cleanedLine.replace(nonWordCharacter, " ");
        }

        return cleanedLine;
    }

    /** Die Zeilen des Textes, ein Zwischenprodukt, welches hier erstellt wird. */
    public List<String> getLines() {
        return lines;
    }

}
