package de.duehl.basics.datetime.date;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.datetime.date.data.IndexAndDate;

public class DateFinderTest {

    @Test
    public void findAllDatesOnlyNormalDate() {
        String text = "20.10.2022";
        DateFinder finder = new DateFinder(text);
        finder.find();

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(1, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        assertEquals(0 , foundDate1.getIndex());
        assertEquals("20.10.2022" , foundDate1.getDate().toString());

        List<String> dates = finder.getDates();
        assertEquals(1, dates.size());
        assertEquals("20.10.2022" , dates.get(0));

        String expectedText = "<<date:1>>";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);
    }

    @Test
    public void findAllDatesOnlyNormalDateSmallMonthAndDay() {
        String text = "2.9.2022";
        DateFinder finder = new DateFinder(text);
        finder.find();

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(1, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        assertEquals(0 , foundDate1.getIndex());
        assertEquals("02.09.2022" , foundDate1.getDate().toString());

        List<String> dates = finder.getDates();
        assertEquals(1, dates.size());
        assertEquals("02.09.2022" , dates.get(0));

        String expectedText = "<<date:1>>";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);
    }

    @Test
    public void findAllDatesOnlyComplicatedDate() {
        String text = "20.10. / 25.11.2022";
        DateFinder finder = new DateFinder(text);
        finder.find();

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(1, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        assertEquals(0 , foundDate1.getIndex());
        assertEquals("20.10.2022" , foundDate1.getDate().toString());

        List<String> dates = finder.getDates();
        assertEquals(1, dates.size());
        assertEquals("20.10.2022" , dates.get(0));

        String expectedText = "<<date:1>>";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);
    }

    @Test
    public void findAllDatesOnlyGermanMonthNameDate() {
        String text = "20. Oktober 2022";
        DateFinder finder = new DateFinder(text);
        finder.find();

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(1, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        assertEquals(0 , foundDate1.getIndex());
        assertEquals("20.10.2022" , foundDate1.getDate().toString());

        List<String> dates = finder.getDates();
        assertEquals(1, dates.size());
        assertEquals("20.10.2022" , dates.get(0));

        String expectedText = "<<date:1>>";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);
    }

    @Test
    public void findAllDates001() {
        String text = "Auf Grund des Verschmelzungsvertrages vom 02.08.2022 und der "
                + "Zustimmungsbeschlüsse vom selben Tage ist die "
                + "Commarco Campus Communications GmbH mit Sitz in Hamburg (Amtsgericht Hamburg, "
                + "HRB 101697) durch Übertragung ihres Vermögens unter Auflösung ohne Abwicklung "
                + "als Ganzes auf die Gesellschaft verschmolzen.";
        DateFinder finder = new DateFinder(text);
        finder.find();

        List<String> dates = finder.getDates();
        assertEquals(1, dates.size());
        assertEquals("02.08.2022" , dates.get(0));

        String expectedText = "Auf Grund des Verschmelzungsvertrages vom <<date:1>> und der "
                + "Zustimmungsbeschlüsse vom selben Tage ist die "
                + "Commarco Campus Communications GmbH mit Sitz in Hamburg (Amtsgericht Hamburg, "
                + "HRB 101697) durch Übertragung ihres Vermögens unter Auflösung ohne Abwicklung "
                + "als Ganzes auf die Gesellschaft verschmolzen.";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(1, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        assertEquals(42 , foundDate1.getIndex());
        assertEquals("02.08.2022" , foundDate1.getDate().toString());
    }

    @Test
    public void findAllDates002() {
        String text = "Die Kulczyk Aircraft 1 GmbH mit dem Sitz in Wien/Österreich "
                + "(Handelsgericht Wien, Firmenbuchnummer 538365p) ist als übertragende "
                + "Gesellschaft auf Grund des Verschmelzungsplans vom 29.08.2022, des "
                + "zustimmenden Beschlusses der Gesellschafterversammlung vom 29.08.2022 und "
                + "der Verschmelzungsbescheinigung vom 06.10.2022 auf die Gesellschaft durch "
                + "Aufnahme verschmolzen.";
        DateFinder finder = new DateFinder(text);
        finder.find();

        String expectedText = "Die Kulczyk Aircraft 1 GmbH mit dem Sitz in Wien/Österreich "
                + "(Handelsgericht Wien, Firmenbuchnummer 538365p) ist als übertragende "
                + "Gesellschaft auf Grund des Verschmelzungsplans vom <<date:1>>, des "
                + "zustimmenden Beschlusses der Gesellschafterversammlung vom <<date:2>> und "
                + "der Verschmelzungsbescheinigung vom <<date:3>> auf die Gesellschaft durch "
                + "Aufnahme verschmolzen.";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);

        List<String> dates = finder.getDates();
        assertEquals(3, dates.size());
        assertEquals("29.08.2022" , dates.get(0));
        assertEquals("29.08.2022" , dates.get(1));
        assertEquals("06.10.2022" , dates.get(2));

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(3, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        IndexAndDate foundDate2 = foundDates.get(1);
        IndexAndDate foundDate3 = foundDates.get(2);
        assertEquals(180 , foundDate1.getIndex());
        assertEquals("29.08.2022" , foundDate1.getDate().toString());
        assertEquals(255 , foundDate2.getIndex());
        assertEquals("29.08.2022" , foundDate2.getDate().toString());
        assertEquals(306 , foundDate3.getIndex());
        assertEquals("06.10.2022" , foundDate3.getDate().toString());
    }

    @Test
    public void findAllDates003() {
        String text = "Die Kulczyk Aircraft 1 GmbH mit dem Sitz in Wien/Österreich "
                + "(Handelsgericht Wien, Firmenbuchnummer 538365p) ist als übertragende "
                + "Gesellschaft auf Grund des Verschmelzungsplans vom 29.8.2022, des "
                + "zustimmenden Beschlusses der Gesellschafterversammlung vom 29.08. / 30.08.2022 "
                + "und der Verschmelzungsbescheinigung vom 06. Oktober 2022 auf die Gesellschaft "
                + "durch Aufnahme verschmolzen.";
        DateFinder finder = new DateFinder(text);
        finder.find();

        String expectedText = "Die Kulczyk Aircraft 1 GmbH mit dem Sitz in Wien/Österreich "
                + "(Handelsgericht Wien, Firmenbuchnummer 538365p) ist als übertragende "
                + "Gesellschaft auf Grund des Verschmelzungsplans vom <<date:2>>, des "
                + "zustimmenden Beschlusses der Gesellschafterversammlung vom <<date:1>> und "
                + "der Verschmelzungsbescheinigung vom <<date:3>> auf die Gesellschaft durch "
                + "Aufnahme verschmolzen.";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);

        List<String> dates = finder.getDates();
        assertEquals(3, dates.size());
        assertEquals("29.08.2022" , dates.get(0));
        assertEquals("29.08.2022" , dates.get(1));
        assertEquals("06.10.2022" , dates.get(2));

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(3, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        IndexAndDate foundDate2 = foundDates.get(1);
        IndexAndDate foundDate3 = foundDates.get(2);
        assertEquals(254, foundDate1.getIndex());
        assertEquals("29.08.2022" , foundDate1.getDate().toString());
        assertEquals(180, foundDate2.getIndex());
        assertEquals("29.08.2022" , foundDate2.getDate().toString());
        assertEquals(306 , foundDate3.getIndex());
        assertEquals("06.10.2022" , foundDate3.getDate().toString());
    }

    @Test
    public void findAllDates004() {
        String text = ""
                + "Die Gesellschaft ist als übertragender Rechtsträger nach Maßgabe des "
                + "Verschmelzungsvertrages vom 19. August 2022 sowie der Zustimmungsbeschlüsse "
                + "ihrer Gesellschafterversammlung vom 19. August 2022 und der "
                + "Gesellschafterversammlung des übernehmenden Rechtsträgers vom 19. August 2022 "
                + "mit der GS Consilium und Commercium GmbH mit Sitz in Mönchengladbach "
                + "(Amtsgericht Mönchengladbach, HRB 20939) verschmolzen. Die Verschmelzung wird "
                + "erst wirksam mit Eintragung auf dem Registerblatt des übernehmenden "
                + "Rechtsträgers.";
        DateFinder finder = new DateFinder(text);
        finder.find();

        String expectedText = ""
                + "Die Gesellschaft ist als übertragender Rechtsträger nach Maßgabe des "
                + "Verschmelzungsvertrages vom <<date:1>> sowie der Zustimmungsbeschlüsse "
                + "ihrer Gesellschafterversammlung vom <<date:2>> und der "
                + "Gesellschafterversammlung des übernehmenden Rechtsträgers vom <<date:3>> "
                + "mit der GS Consilium und Commercium GmbH mit Sitz in Mönchengladbach "
                + "(Amtsgericht Mönchengladbach, HRB 20939) verschmolzen. Die Verschmelzung wird "
                + "erst wirksam mit Eintragung auf dem Registerblatt des übernehmenden "
                + "Rechtsträgers.";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);

        List<String> dates = finder.getDates();
        assertEquals(3, dates.size());
        assertEquals("19.08.2022" , dates.get(0));
        assertEquals("19.08.2022" , dates.get(1));
        assertEquals("19.08.2022" , dates.get(2));

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(3, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        IndexAndDate foundDate2 = foundDates.get(1);
        IndexAndDate foundDate3 = foundDates.get(2);
        assertEquals(97, foundDate1.getIndex());
        assertEquals("19.08.2022" , foundDate1.getDate().toString());
        assertEquals(176, foundDate2.getIndex());
        assertEquals("19.08.2022" , foundDate2.getDate().toString());
        assertEquals(257 , foundDate3.getIndex());
        assertEquals("19.08.2022" , foundDate3.getDate().toString());
    }

    @Test
    public void findAllDates005() {
        String text = ""
                + "Die Gesellschaft ist als übertragender Rechtsträger nach Maßgabe des "
                + "Verschmelzungsvertrages vom 14.06.2022 nebst Ergänzung vom 19.10.2022 sowie "
                + "der Zustimmungsbeschlüsse ihrer Gesellschafterversammlung vom 14.06. und "
                + "19.10.2022 und der Gesellschafterversammlung des übernehmenden Rechtsträgers "
                + "vom 14.06. und 19.10.2022 mit der Coppenrath Beteiligungs GmbH mit Sitz in "
                + "Steinfurt (Amtsgericht Steinfurt HRB 10146) verschmolzen.";
        DateFinder finder = new DateFinder(text);
        finder.find();

        String expectedText = ""
                + "Die Gesellschaft ist als übertragender Rechtsträger nach Maßgabe des "
                + "Verschmelzungsvertrages vom <<date:3>> nebst Ergänzung vom <<date:4>> sowie "
                + "der Zustimmungsbeschlüsse ihrer Gesellschafterversammlung vom <<date:1>> und "
                + "der Gesellschafterversammlung des übernehmenden Rechtsträgers "
                + "vom <<date:2>> mit der Coppenrath Beteiligungs GmbH mit Sitz in "
                + "Steinfurt (Amtsgericht Steinfurt HRB 10146) verschmolzen.";
        String actualText = finder.getText();
        assertEquals(expectedText, actualText);

        List<String> dates = finder.getDates();
        assertEquals(4, dates.size());
        assertEquals("14.06.2022" , dates.get(0)); // nur das erste Datum
        assertEquals("14.06.2022" , dates.get(1)); // nur das erste Datum
        assertEquals("14.06.2022" , dates.get(2));
        assertEquals("19.10.2022" , dates.get(3));

        List<IndexAndDate> foundDates = finder.getFoundDates();
        assertEquals(4, foundDates.size());
        IndexAndDate foundDate1 = foundDates.get(0);
        IndexAndDate foundDate2 = foundDates.get(1);
        IndexAndDate foundDate3 = foundDates.get(2);
        IndexAndDate foundDate4 = foundDates.get(3);
        assertEquals(207, foundDate1.getIndex());
        assertEquals("14.06.2022" , foundDate1.getDate().toString());
        assertEquals(288, foundDate2.getIndex());
        assertEquals("14.06.2022" , foundDate2.getDate().toString());
        assertEquals(97 , foundDate3.getIndex());
        assertEquals("14.06.2022" , foundDate3.getDate().toString());
        assertEquals(128 , foundDate4.getIndex());
        assertEquals("19.10.2022" , foundDate4.getDate().toString());
    }

}
