package de.duehl.basics.datetime.time;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.junit.Test;

public class TimeHelperTest {

    @Test
    public void secondsToHoursMinutesSeconds1() {
        long seconds = 51;
        String expedted = "00:00:51";
        String actual = TimeHelper.secondsToHoursMinutesSeconds(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSeconds2() {
        long seconds = 59;
        String expedted = "00:00:59";
        String actual = TimeHelper.secondsToHoursMinutesSeconds(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSeconds3() {
        long seconds = 60;
        String expedted = "00:01:00";
        String actual = TimeHelper.secondsToHoursMinutesSeconds(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSeconds4() {
        long seconds = 119;
        String expedted = "00:01:59";
        String actual = TimeHelper.secondsToHoursMinutesSeconds(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSeconds5() {
        long seconds = 27 * 60 * 60  +  12 * 60  + 22;
        String expedted = "27:12:22";
        String actual = TimeHelper.secondsToHoursMinutesSeconds(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSecondsSinceMidnight1() {
        long seconds = 51;
        String expedted = "00:00:51";
        String actual = TimeHelper.secondsToHoursMinutesSecondsSinceMidnight(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSecondsSinceMidnight2() {
        long seconds = 59;
        String expedted = "00:00:59";
        String actual = TimeHelper.secondsToHoursMinutesSecondsSinceMidnight(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSecondsSinceMidnight3() {
        long seconds = 60;
        String expedted = "00:01:00";
        String actual = TimeHelper.secondsToHoursMinutesSecondsSinceMidnight(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSecondsSinceMidnight4() {
        long seconds = 119;
        String expedted = "00:01:59";
        String actual = TimeHelper.secondsToHoursMinutesSecondsSinceMidnight(seconds);
        assertEquals(actual, expedted);
    }

    @Test
    public void secondsToHoursMinutesSecondsSinceMidnight5() {
        long seconds = 27 * 60 * 60  +  12 * 60  + 22;
        String expedted = "03:12:22";
        String actual = TimeHelper.secondsToHoursMinutesSecondsSinceMidnight(seconds);
        assertEquals(actual, expedted);
    }

    private class SecondsAndHumanReadable {
        private final long seconds;
        private final String hoursMinutesSeconds;
        public SecondsAndHumanReadable(long seconds, String hoursMinutesSeconds) {
            this.seconds = seconds;
            this.hoursMinutesSeconds = hoursMinutesSeconds;
        }
        public long getSeconds() {
            return seconds;
        }
        public String getHoursMinutesSeconds() {
            return hoursMinutesSeconds;
        }
    }

    @Test
    public void hoursMinutesSecondsToSecondsMany() {
        List<SecondsAndHumanReadable> sahrs = new ArrayList<>();
        sahrs.add(new SecondsAndHumanReadable(        0, "00:00:00"));
        sahrs.add(new SecondsAndHumanReadable(        1, "00:00:01"));
        sahrs.add(new SecondsAndHumanReadable(       36, "00:00:36"));
        sahrs.add(new SecondsAndHumanReadable(       51, "00:00:51"));
        sahrs.add(new SecondsAndHumanReadable(       59, "00:00:59"));
        sahrs.add(new SecondsAndHumanReadable(       60, "00:01:00"));
        sahrs.add(new SecondsAndHumanReadable(      119, "00:01:59"));
        sahrs.add(new SecondsAndHumanReadable(      120, "00:02:00"));
        sahrs.add(new SecondsAndHumanReadable(      121, "00:02:01"));
        sahrs.add(new SecondsAndHumanReadable(      179, "00:02:59"));
        sahrs.add(new SecondsAndHumanReadable(      180, "00:03:00"));
        sahrs.add(new SecondsAndHumanReadable(      299, "00:04:59"));
        sahrs.add(new SecondsAndHumanReadable(      300, "00:05:00"));
        sahrs.add(new SecondsAndHumanReadable(      360, "00:06:00"));
        sahrs.add(new SecondsAndHumanReadable(      600, "00:10:00"));
        sahrs.add(new SecondsAndHumanReadable(     1200, "00:20:00"));
        sahrs.add(new SecondsAndHumanReadable(     1800, "00:30:00"));
        sahrs.add(new SecondsAndHumanReadable(     2400, "00:40:00"));
        sahrs.add(new SecondsAndHumanReadable(     3000, "00:50:00"));
        sahrs.add(new SecondsAndHumanReadable(     3599, "00:59:59"));
        sahrs.add(new SecondsAndHumanReadable(     3600, "01:00:00"));
        sahrs.add(new SecondsAndHumanReadable(     7200, "02:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    10800, "03:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    36000, "10:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    86400, "24:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    86401, "24:00:01"));
        sahrs.add(new SecondsAndHumanReadable(    90000, "25:00:00"));
        sahrs.add(new SecondsAndHumanReadable(   180000, "50:00:00"));
        sahrs.add(new SecondsAndHumanReadable(   183600, "51:00:00"));
        sahrs.add(new SecondsAndHumanReadable(   359999, "99:59:59"));
        sahrs.add(new SecondsAndHumanReadable(   360000, "100:00:00"));
        sahrs.add(new SecondsAndHumanReadable(  3600000, "1000:00:00"));
        sahrs.add(new SecondsAndHumanReadable( 36000000, "10000:00:00"));
        sahrs.add(new SecondsAndHumanReadable(360000000, "100000:00:00"));
        sahrs.add(new SecondsAndHumanReadable(360000001, "100000:00:01"));

        for (SecondsAndHumanReadable sahr : sahrs) {
            String hoursMinutesSeconds = sahr.getHoursMinutesSeconds();
            long expected = sahr.getSeconds();
            long actual = TimeHelper.hoursMinutesSecondsToSeconds(hoursMinutesSeconds);
            assertEquals(expected, actual);
        }
    }

    @Test (expected = IllegalArgumentException.class)
    public void hoursMinutesSecondsToSecondsFailing1() {
        String hoursMinutesSeconds = "000000";
        TimeHelper.hoursMinutesSecondsToSeconds(hoursMinutesSeconds);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hoursMinutesSecondsToSecondsFailing2() {
        String hoursMinutesSeconds = "00:00:00:00";
        TimeHelper.hoursMinutesSecondsToSeconds(hoursMinutesSeconds);
    }

    @Test (expected = IllegalArgumentException.class)
    public void hoursMinutesSecondsToSecondsFailing3() {
        String hoursMinutesSeconds = "00:OO:00";
        TimeHelper.hoursMinutesSecondsToSeconds(hoursMinutesSeconds);
    }

    @Test
    public void secondsToHoursMinutesSecondsMany() {
        List<SecondsAndHumanReadable> sahrs = new ArrayList<>();
        sahrs.add(new SecondsAndHumanReadable(       -1, "ERROR"));
        sahrs.add(new SecondsAndHumanReadable(        0, "00:00:00"));
        sahrs.add(new SecondsAndHumanReadable(        1, "00:00:01"));
        sahrs.add(new SecondsAndHumanReadable(       36, "00:00:36"));
        sahrs.add(new SecondsAndHumanReadable(       51, "00:00:51"));
        sahrs.add(new SecondsAndHumanReadable(       59, "00:00:59"));
        sahrs.add(new SecondsAndHumanReadable(       60, "00:01:00"));
        sahrs.add(new SecondsAndHumanReadable(      119, "00:01:59"));
        sahrs.add(new SecondsAndHumanReadable(      120, "00:02:00"));
        sahrs.add(new SecondsAndHumanReadable(      121, "00:02:01"));
        sahrs.add(new SecondsAndHumanReadable(      179, "00:02:59"));
        sahrs.add(new SecondsAndHumanReadable(      180, "00:03:00"));
        sahrs.add(new SecondsAndHumanReadable(      299, "00:04:59"));
        sahrs.add(new SecondsAndHumanReadable(      300, "00:05:00"));
        sahrs.add(new SecondsAndHumanReadable(      360, "00:06:00"));
        sahrs.add(new SecondsAndHumanReadable(      600, "00:10:00"));
        sahrs.add(new SecondsAndHumanReadable(     1200, "00:20:00"));
        sahrs.add(new SecondsAndHumanReadable(     1800, "00:30:00"));
        sahrs.add(new SecondsAndHumanReadable(     2400, "00:40:00"));
        sahrs.add(new SecondsAndHumanReadable(     3000, "00:50:00"));
        sahrs.add(new SecondsAndHumanReadable(     3599, "00:59:59"));
        sahrs.add(new SecondsAndHumanReadable(     3600, "01:00:00"));
        sahrs.add(new SecondsAndHumanReadable(     7200, "02:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    10800, "03:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    36000, "10:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    86400, "24:00:00"));
        sahrs.add(new SecondsAndHumanReadable(    86401, "24:00:01"));
        sahrs.add(new SecondsAndHumanReadable(    90000, "25:00:00"));
        sahrs.add(new SecondsAndHumanReadable(   180000, "50:00:00"));
        sahrs.add(new SecondsAndHumanReadable(   183600, "51:00:00"));
        sahrs.add(new SecondsAndHumanReadable(   359999, "99:59:59"));
        sahrs.add(new SecondsAndHumanReadable(   360000, "100:00:00"));
        sahrs.add(new SecondsAndHumanReadable(  3600000, "1000:00:00"));
        sahrs.add(new SecondsAndHumanReadable( 36000000, "10000:00:00"));
        sahrs.add(new SecondsAndHumanReadable(360000000, "100000:00:00"));
        sahrs.add(new SecondsAndHumanReadable(360000001, "100000:00:01"));

        for (SecondsAndHumanReadable sahr : sahrs) {
            long seconds = sahr.getSeconds();
            String expected = sahr.getHoursMinutesSeconds();
            String actual = TimeHelper.secondsToHoursMinutesSeconds(seconds);
            assertEquals(expected, actual);
        }
    }

    @Test
    public void milliSecondsToHoursMinutesSecondsMilliSecondsMany() {
        List<SecondsAndHumanReadable> sahrs = new ArrayList<>();

        sahrs.add(new SecondsAndHumanReadable(       -1, "ERROR"));
        sahrs.add(new SecondsAndHumanReadable(        0, "00:00:00:000"));
        sahrs.add(new SecondsAndHumanReadable(        1, "00:00:00:001"));
        sahrs.add(new SecondsAndHumanReadable(       36, "00:00:00:036"));
        sahrs.add(new SecondsAndHumanReadable(       99, "00:00:00:099"));
        sahrs.add(new SecondsAndHumanReadable(      100, "00:00:00:100"));
        sahrs.add(new SecondsAndHumanReadable(      999, "00:00:00:999"));
        sahrs.add(new SecondsAndHumanReadable(     1000, "00:00:01:000"));
        sahrs.add(new SecondsAndHumanReadable(     1999, "00:00:01:999"));
        sahrs.add(new SecondsAndHumanReadable(    59999, "00:00:59:999"));
        sahrs.add(new SecondsAndHumanReadable(    60000, "00:01:00:000"));

        for (SecondsAndHumanReadable sahr : sahrs) {
            long milliSeconds = sahr.getSeconds();
            String expected = sahr.getHoursMinutesSeconds();
            String actual = TimeHelper.milliSecondsToHoursMinutesSecondsMilliSeconds(
                    milliSeconds);
            assertEquals("Eingabe : " + milliSeconds, expected, actual);
        }
    }

    @Test
    public void milliSecondsToHoursMinutesSecondsMilliSecondsBracesMany() {
        List<SecondsAndHumanReadable> sahrs = new ArrayList<>();

        sahrs.add(new SecondsAndHumanReadable(       -1, "ERROR"));
        sahrs.add(new SecondsAndHumanReadable(        0, "00:00:00(000)"));
        sahrs.add(new SecondsAndHumanReadable(        1, "00:00:00(001)"));
        sahrs.add(new SecondsAndHumanReadable(       36, "00:00:00(036)"));
        sahrs.add(new SecondsAndHumanReadable(       99, "00:00:00(099)"));
        sahrs.add(new SecondsAndHumanReadable(      100, "00:00:00(100)"));
        sahrs.add(new SecondsAndHumanReadable(      999, "00:00:00(999)"));
        sahrs.add(new SecondsAndHumanReadable(     1000, "00:00:01(000)"));
        sahrs.add(new SecondsAndHumanReadable(     1999, "00:00:01(999)"));
        sahrs.add(new SecondsAndHumanReadable(    59999, "00:00:59(999)"));
        sahrs.add(new SecondsAndHumanReadable(    60000, "00:01:00(000)"));

        for (SecondsAndHumanReadable sahr : sahrs) {
            long milliSeconds = sahr.getSeconds();
            String expected = sahr.getHoursMinutesSeconds();
            String actual = TimeHelper.milliSecondsToHoursMinutesSecondsMilliSecondsBraces(
                    milliSeconds);
            assertEquals("Eingabe : " + milliSeconds, expected, actual);
        }
    }

    @Test
    public void calendarToTimeTestNow() {
        Calendar cal = Calendar.getInstance();
        ImmutualTime time = TimeHelper.calendarToTime(cal);
        assertNotNull(time);
    }

    @Test
    public void calendarToTimeTestConcreteDate() {
        Calendar cal = Calendar.getInstance();
        cal.set(2016, 10 - 1, 12, 11, 25, 0);

        ImmutualTime time = TimeHelper.calendarToTime(cal);

        int actualHour = time.getHour();
        int actualMinute = time.getMinute();
        int actualSecond = time.getSecond();

        int expectedHour = 11;
        int expectedMinute = 25;
        int expectedSecond = 0;

        assertEquals(expectedHour, actualHour);
        assertEquals(expectedMinute, actualMinute);
        assertEquals(expectedSecond, actualSecond);
    }

    @Test
    public void isHhMmTime01() {
        String time = "00:00";
        boolean actual = TimeHelper.isHhMmTime(time);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void isHhMmTime02() {
        String time = "00;00";
        boolean actual = TimeHelper.isHhMmTime(time);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void isHhMmTime03() {
        String time = "0000";
        boolean actual = TimeHelper.isHhMmTime(time);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void isHhMmTime04() {
        String time = "00000";
        boolean actual = TimeHelper.isHhMmTime(time);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void isHhMmTime05() {
        String time = "24:00";
        boolean actual = TimeHelper.isHhMmTime(time);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void isHhMmTime06() {
        String time = "23:60";
        boolean actual = TimeHelper.isHhMmTime(time);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test01() {
        String digit = "0";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test02() {
        String digit = "1";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test03() {
        String digit = "2";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test04() {
        String digit = "";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test05() {
        String digit = "3";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test06() {
        String digit = "4";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test07() {
        String digit = "5";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test08() {
        String digit = "6";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test09() {
        String digit = "7";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test10() {
        String digit = "8";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0or1or2Test11() {
        String digit = "9";
        boolean actual = TimeHelper.is0or1or2(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test01() {
        String digit = "0";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test02() {
        String digit = "1";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test03() {
        String digit = "2";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test04() {
        String digit = "";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test05() {
        String digit = "3";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test06() {
        String digit = "4";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test07() {
        String digit = "5";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test08() {
        String digit = "6";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test09() {
        String digit = "7";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test10() {
        String digit = "8";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to3Test11() {
        String digit = "9";
        boolean actual = TimeHelper.is0to3(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test01() {
        String digit = "0";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test02() {
        String digit = "1";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test03() {
        String digit = "2";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test04() {
        String digit = "";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test05() {
        String digit = "3";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test06() {
        String digit = "4";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test07() {
        String digit = "5";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test08() {
        String digit = "6";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test09() {
        String digit = "7";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test10() {
        String digit = "8";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to5Test11() {
        String digit = "9";
        boolean actual = TimeHelper.is0to5(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test01() {
        String digit = "0";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test02() {
        String digit = "1";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test03() {
        String digit = "2";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test04() {
        String digit = "";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = false;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test05() {
        String digit = "3";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test06() {
        String digit = "4";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test07() {
        String digit = "5";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test08() {
        String digit = "6";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test09() {
        String digit = "7";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test10() {
        String digit = "8";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void is0to9Test11() {
        String digit = "9";
        boolean actual = TimeHelper.is0to9(digit);
        boolean expected = true;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsEmptyInput() {
        String runtime = "";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = -1L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsNotEnoughParts() {
        String runtime = "12:34";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = -1L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsTooMuchParts() {
        String runtime = "12:34:56:78";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = -1L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsParsingFailureHour() {
        String runtime = "ab:34:56";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = -1L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsParsingFailureMinute() {
        String runtime = "12:ab:56";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = -1L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsParsingFailureSecond() {
        String runtime = "12:34:ab";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = -1L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsAFewSeconds() {
        String runtime = "00:00:12";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = 12L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsAFewMinutes() {
        String runtime = "00:12:00";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = 12L * 60L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSecondsAFewHours() {
        String runtime = "15:00:00";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = 15L * 60L * 60L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSeconds235959() {
        String runtime = "23:59:59";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = 23L * 60L * 60L + 59L * 60L + 59L;
        assertEquals(expected, actual);
    }

    @Test
    public void runtimeToSeconds12235959() {
        String runtime = "1223:59:59";
        long actual = TimeHelper.runtimeToSeconds(runtime);
        long expected = 1223L * 60L * 60L + 59L * 60L + 59L;
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntimeZeroInput() {
        long seconds = 0;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "00:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntimeNegativeInput() {
        long seconds = -1;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "unbekannt";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntime13() {
        long seconds = 13;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "00:00:13";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntime61() {
        long seconds = 61;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "00:01:01";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntime3599() {
        long seconds = 3599;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "00:59:59";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntime3600() {
        long seconds = 3600;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "01:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntime36000() {
        long seconds = 36000;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "10:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntime39600() {
        long seconds = 39600;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "11:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void secondsToRuntime39661() {
        long seconds = 39661;
        String actual = TimeHelper.secondsToRuntime(seconds);
        String expected = "11:01:01";
        assertEquals(expected, actual);
    }

    @Test
    public void summerizeHourMinuteSeconds01() {
        String actual = TimeHelper.summerizeHourMinuteSeconds("01:00:00", "01:00:00");
        String expected = "02:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void summerizeHourMinuteSeconds02() {
        String actual = TimeHelper.summerizeHourMinuteSeconds("01:00:00", "00:59:59");
        String expected = "01:59:59";
        assertEquals(expected, actual);
    }

    @Test
    public void summerizeHourMinuteSeconds03() {
        String actual = TimeHelper.summerizeHourMinuteSeconds("01:00:00", "00:59:59", "200:00:01");
        String expected = "202:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void addHourMinutesSecondsToHourMinutesSeconds0and0() {
        String time1 = "00:00:00";
        String time2 = "00:00:00";
        String actual = TimeHelper.addHourMinutesSecondsToHourMinutesSeconds(time1, time2);
        String expected = "00:00:00";
        assertEquals(expected, actual);
    }

    @Test
    public void addHourMinutesSecondsToHourMinutesSeconds60and0() {
        String time1 = "00:01:00";
        String time2 = "00:00:00";
        String actual = TimeHelper.addHourMinutesSecondsToHourMinutesSeconds(time1, time2);
        String expected = "00:01:00";
        assertEquals(expected, actual);
    }

    @Test
    public void addHourMinutesSecondsToHourMinutesSeconds60and60() {
        String time1 = "00:01:00";
        String time2 = "00:01:00";
        String actual = TimeHelper.addHourMinutesSecondsToHourMinutesSeconds(time1, time2);
        String expected = "00:02:00";
        assertEquals(expected, actual);
    }

    @Test
    public void addHourMinutesSecondsToHourMinutesSeconds119and61() {
        String time1 = "00:01:59";
        String time2 = "00:01:01";
        String actual = TimeHelper.addHourMinutesSecondsToHourMinutesSeconds(time1, time2);
        String expected = "00:03:00";
        assertEquals(expected, actual);
    }

    @Test
    public void addHourMinutesSecondsToHourMinutesSeconds() {
        String time1 = "04:24:25";
        String time2 = "02:59:13";
        String actual = TimeHelper.addHourMinutesSecondsToHourMinutesSeconds(time1, time2);
        String expected = "07:23:38";
        assertEquals(expected, actual);
    }

    @Test
    public void insertTimeColonsHHMMSS() {
        String dateWithoutDots = "131742";
        String actual = TimeHelper.insertTimeColonsHHMMSS(dateWithoutDots);
        String expected = "13:17:42";
        assertEquals(expected , actual);
    }

    @Test
    public void insertTimeColonsHHMMSSWrongLength() {
        String dateWithoutDots = "1317420";
        String actual = TimeHelper.insertTimeColonsHHMMSS(dateWithoutDots);
        String expected = dateWithoutDots;
        assertEquals(expected , actual);
    }

    @Test
    public void insertTimeColonsHHMMSSNotOnlyDigits() {
        String dateWithoutDots = "13I742";
        String actual = TimeHelper.insertTimeColonsHHMMSS(dateWithoutDots);
        String expected = dateWithoutDots;
        assertEquals(expected , actual);
    }

    @Test
    public void secondsToMinutesSeconds() {
        int seconds = 130;
        String actual = TimeHelper.secondsToMinutesSeconds(seconds);
        String expected = "2:10";
        assertEquals(expected , actual);
    }

}
