package de.duehl.basics.text;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;
import static de.duehl.basics.text.Text.LINE_BREAK;

import org.junit.Test;

public class TextBreakerTest {

    @Test
    public void breakNoBreak() {
        String text = "Mal sehen was so passiert!";
        String expected = text;
        int lineLength = 80;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakNoBreakKeepSpacesAtBreaks() {
        String text = "Mal sehen was so passiert!";
        String expected = text;
        int lineLength = 80;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakNoBreakNotRemovingTrailingBlank() {
        String text = "Mal sehen was so passiert! ";
        String expected = "Mal sehen was so passiert! ";
        int lineLength = 80;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakNoBreakNotRemovingTrailingBlankKeepSpacesAtBreaks() {
        String text = "Mal sehen was so passiert! ";
        String expected = "Mal sehen was so passiert! ";
        int lineLength = 80;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddOneBreak() {
        String text = "Mal sehen was so passiert!";
        String expected = "Mal sehen was" + LINE_BREAK
                + "so passiert!";
        int lineLength = 15;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddOneBreakKeepSpacesAtBreaks() {
        String text = "Mal sehen was so passiert!";
        String expected = "Mal sehen was " + LINE_BREAK
                + "so passiert!";
        int lineLength = 15;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddFourBreaks() {
        String text = "Mal sehen was so passiert!";
        String expected = "Mal" + LINE_BREAK
                + "sehen" + LINE_BREAK
                + "was" + LINE_BREAK
                + "so" + LINE_BREAK
                + "passiert!";
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddFourBreaksKeepSpacesAtBreaks() {
        String text = "Mal sehen was so passiert!";
        String expected = "Mal " + LINE_BREAK
                + "sehen " + LINE_BREAK
                + "was " + LINE_BREAK
                + "so " + LINE_BREAK
                + "passiert!";
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddBreaksWithLongParts() {
        String text = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        String expected = "Mal" + LINE_BREAK
                + "sehen" + LINE_BREAK
                + "was" + LINE_BREAK
                + "so" + LINE_BREAK
                + "AUSSERGEWÖHNLICHLANGES" + LINE_BREAK
                + "passiert!";
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddBreaksWithLongPartsKeepSpacesAtBreaks() {
        String text = "Mal sehen was so AUSSERGEWÖHNLICHLANGES passiert!";
        String expected = "Mal " + LINE_BREAK
                + "sehen " + LINE_BREAK
                + "was " + LINE_BREAK
                + "so " + LINE_BREAK
                + "AUSSERGEWÖHNLICHLANGES " + LINE_BREAK
                + "passiert!";
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddNoBreaksAtUnderscores() {
        String text = "Mal_sehen_was_so_AUSSERGEWÖHNLICHLANGES_passiert!";
        String expected = text;
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAddNoBreaksAtUnderscoresKeepSpacesAtBreaks() {
        String text = "Mal_sehen_was_so_AUSSERGEWÖHNLICHLANGES_passiert!";
        String expected = text;
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBreakAtExaktTextPartLengths() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK
                + "cd" + LINE_BREAK
                + "ef";
        int lineLength = 2;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBreakAtExaktTextPartLengthsKeepSpacesAtBreaks() {
        String text = "ab cd ef";
        String expected = "ab " + LINE_BREAK
                + "cd " + LINE_BREAK
                + "ef";
        int lineLength = 2;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBreakAtSmallerLineLengthThanTextPartLength() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK
                + "cd" + LINE_BREAK
                + "ef";
        int lineLength = 1;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBreakAtSmallerLineLengthThanTextPartLengthKeepSpacesAtBreaks() {
        String text = "ab cd ef";
        String expected = "ab " + LINE_BREAK
                + "cd " + LINE_BREAK
                + "ef";
        int lineLength = 1;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBreakAtLineLengthEqualsTextPartLengthAndSpace() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK
                + "cd" + LINE_BREAK
                + "ef";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBreakAtLineLengthEqualsTextPartLengthAndSpaceKeepSpacesAtBreaks() {
        String text = "ab cd ef";
        String expected = "ab " + LINE_BREAK
                + "cd " + LINE_BREAK
                + "ef";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBeforeLineIsTooLong() {
        String text = "ab cd ef";
        String expected = "ab" + LINE_BREAK
                + "cd" + LINE_BREAK
                + "ef";
        int lineLength = 4;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakBeforeLineIsTooLongKeepSpacesAtBreaks() {
        String text = "ab cd ef";
        String expected = "ab " + LINE_BREAK
                + "cd " + LINE_BREAK
                + "ef";
        int lineLength = 4;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAfterTwoParts() {
        String text = "ab cd ef";
        String expected = "ab cd" + LINE_BREAK
                + "ef";
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAfterTwoPartsTooShortWithSpaceBehindKeepSpacesAtBreaks() {
        String text = "ab cd ef";
        String expected = "ab " + LINE_BREAK
                + "cd ef";
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakAfterTwoPartsKeepSpacesAtBreaks() {
        String text = "ab cd ef";
        String expected = "ab cd " + LINE_BREAK
                + "ef";
        int lineLength = 6;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakSmallParts() {
        String text = "ab cd ef gh ij kl mn op qr";
        String expected = "ab cd" + LINE_BREAK
                + "ef gh" + LINE_BREAK
                + "ij kl" + LINE_BREAK
                + "mn op" + LINE_BREAK
                + "qr";
        int lineLength = 5;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakSmallPartsKeepSpacesAtBreaks() {
        String text = "ab cd ef gh ij kl mn op qr";
        String expected = "ab cd " + LINE_BREAK
                + "ef gh " + LINE_BREAK
                + "ij kl " + LINE_BREAK
                + "mn op " + LINE_BREAK
                + "qr";
        int lineLength = 6;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakHandleExistingBreakes() {
        String text = "Mal\nsehen\r\nwas so AUSSERGEWÖHNLICHLANGES passiert!";
        String expected = "Mal" + LINE_BREAK
                + "sehen" + LINE_BREAK
                + "was so" + LINE_BREAK
                + "AUSSERGEWÖHNLICHLANGES" + LINE_BREAK
                + "passiert!";
        int lineLength = 10;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakHandleExistingBreakesKeepSpacesAtBreaks() {
        String text = "Mal\nsehen\r\nwas so AUSSERGEWÖHNLICHLANGES passiert!";
        String expected = "Mal" + LINE_BREAK
                + "sehen" + LINE_BREAK
                + "was so " + LINE_BREAK
                + "AUSSERGEWÖHNLICHLANGES " + LINE_BREAK
                + "passiert!";
        int lineLength = 10;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces() {
        String text = "ab" + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab" + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 2;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpacesKeepSpacesAtBreaks() {
        String text = "ab" + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab" + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 2;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces2() {
        String text = "ab ab " + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab" + LINE_BREAK
                + "ab " + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces2KeepSpacesAtBreaks() {
        String text = "ab ab " + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab " + LINE_BREAK
                + "ab " + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces3() {
        String text = "ab ab" + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab" + LINE_BREAK
                + "ab" + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces3KeepSpacesAtBreaks() {
        String text = "ab ab" + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab " + LINE_BREAK
                + "ab" + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces4() {
        String text = "ab x" + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab" + LINE_BREAK
                + "x" + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces4KeepSpacesAtBreaks() {
        String text = "ab x" + LINE_BREAK + " " + LINE_BREAK + "cd";
        String expected = "ab " + LINE_BREAK
                + "x" + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces5() {
        String text = "ab " + LINE_BREAK;
        String expected = "ab " + LINE_BREAK;
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakExistingBreaksWithSpaces5KeepSpacesAtBreaks() {
        String text = "ab " + LINE_BREAK;
        String expected = "ab " + LINE_BREAK;
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void multipleSpaces1() {
        String text = "ab   cd";
        String expected = "ab " + LINE_BREAK
                + " cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void multipleSpaces1KeepSpacesAtBreaks() {
        String text = "ab   cd";
        String expected = "ab " + LINE_BREAK
                + "  " + LINE_BREAK
                + "cd";
        int lineLength = 3;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void multipleSpaces2() {
        String text = "ab   cd";
        String expected = "ab" + LINE_BREAK
                + " " + LINE_BREAK
                + "cd";
        int lineLength = 2;
        TextBreaker breaker = new TextBreaker(lineLength);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void multipleSpaces2KeepSpacesAtBreaks() {
        String text = "ab   cd";
        String expected = "ab " + LINE_BREAK
                + "  " + LINE_BREAK
                + "cd";
        int lineLength = 2;
        TextBreaker breaker = new TextBreaker(lineLength);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakWithFrontSpaces() {
        String text = "Lorem ipsum dolor sit amet, consectetur adipisici elit, sed eiusmod tempor "
                + "incidunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis "
                + "nostrud exercitation ullamco laboris nisi ut aliquid ex ea commodi consequat. "
                + "Quis aute iure reprehenderit in voluptate velit esse cillum dolore eu fugiat "
                + "nulla pariatur. Excepteur sint obcaecat cupiditat non proident, sunt in culpa "
                + "qui officia deserunt mollit anim id est laborum.";
        String expected = "Lorem ipsum" + LINE_BREAK
                + "     dolor sit" +  LINE_BREAK
                + "     amet," +  LINE_BREAK
                + "     consectetur" +  LINE_BREAK
                + "     adipisici" +  LINE_BREAK
                + "     elit, sed" +  LINE_BREAK
                + "     eiusmod" +  LINE_BREAK
                + "     tempor" +  LINE_BREAK
                + "     incidunt" +  LINE_BREAK
                + "     ut labore" +  LINE_BREAK
                + "     et dolore" +  LINE_BREAK
                + "     magna" +  LINE_BREAK
                + "     aliqua. Ut" +  LINE_BREAK
                + "     enim ad" +  LINE_BREAK
                + "     minim" +  LINE_BREAK
                + "     veniam," +  LINE_BREAK
                + "     quis" +  LINE_BREAK
                + "     nostrud" +  LINE_BREAK
                + "     exercitation" +  LINE_BREAK
                + "     ullamco" +  LINE_BREAK
                + "     laboris" +  LINE_BREAK
                + "     nisi ut" +  LINE_BREAK
                + "     aliquid ex" +  LINE_BREAK
                + "     ea commodi" +  LINE_BREAK
                + "     consequat." +  LINE_BREAK
                + "     Quis aute" +  LINE_BREAK
                + "     iure" +  LINE_BREAK
                + "     reprehenderit" +  LINE_BREAK
                + "     in" +  LINE_BREAK
                + "     voluptate" +  LINE_BREAK
                + "     velit esse" +  LINE_BREAK
                + "     cillum" +  LINE_BREAK
                + "     dolore eu" +  LINE_BREAK
                + "     fugiat" +  LINE_BREAK
                + "     nulla" +  LINE_BREAK
                + "     pariatur." +  LINE_BREAK
                + "     Excepteur" +  LINE_BREAK
                + "     sint" +  LINE_BREAK
                + "     obcaecat" +  LINE_BREAK
                + "     cupiditat" +  LINE_BREAK
                + "     non" +  LINE_BREAK
                + "     proident," +  LINE_BREAK
                + "     sunt in" +  LINE_BREAK
                + "     culpa qui" +  LINE_BREAK
                + "     officia" +  LINE_BREAK
                + "     deserunt" +  LINE_BREAK
                + "     mollit" +  LINE_BREAK
                + "     anim id" +  LINE_BREAK
                + "     est" +  LINE_BREAK
                + "     laborum.";
        int numberOfFrontSpaces = 5;
        int lineLength = 15;
        TextBreaker breaker = new TextBreaker(lineLength, numberOfFrontSpaces);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void breakWithFrontSpacesKeepSpacesAtBreaks() {
        String text = "Lorem ipsum dolor sit amet, consectetur adipisici elit, sed eiusmod tempor "
                + "incidunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis "
                + "nostrud exercitation ullamco laboris nisi ut aliquid ex ea commodi consequat. "
                + "Quis aute iure reprehenderit in voluptate velit esse cillum dolore eu fugiat "
                + "nulla pariatur. Excepteur sint obcaecat cupiditat non proident, sunt in culpa "
                + "qui officia deserunt mollit anim id est laborum.";
        String expected = "Lorem ipsum " +  LINE_BREAK
                + "     dolor sit " +  LINE_BREAK
                + "     amet, " +  LINE_BREAK
                + "     consectetur " +  LINE_BREAK
                + "     adipisici " +  LINE_BREAK
                + "     elit, sed " +  LINE_BREAK
                + "     eiusmod " +  LINE_BREAK
                + "     tempor " +  LINE_BREAK
                + "     incidunt " +  LINE_BREAK
                + "     ut labore " +  LINE_BREAK
                + "     et dolore " +  LINE_BREAK
                + "     magna " +  LINE_BREAK
                + "     aliqua. " +  LINE_BREAK
                + "     Ut enim " +  LINE_BREAK
                + "     ad minim " +  LINE_BREAK
                + "     veniam, " +  LINE_BREAK
                + "     quis " +  LINE_BREAK
                + "     nostrud " +  LINE_BREAK
                + "     exercitation " +  LINE_BREAK
                + "     ullamco " +  LINE_BREAK
                + "     laboris " +  LINE_BREAK
                + "     nisi ut " +  LINE_BREAK
                + "     aliquid " +  LINE_BREAK
                + "     ex ea " +  LINE_BREAK
                + "     commodi " +  LINE_BREAK
                + "     consequat. " +  LINE_BREAK
                + "     Quis aute " +  LINE_BREAK
                + "     iure " +  LINE_BREAK
                + "     reprehenderit " +  LINE_BREAK
                + "     in " +  LINE_BREAK
                + "     voluptate " +  LINE_BREAK
                + "     velit " +  LINE_BREAK
                + "     esse " +  LINE_BREAK
                + "     cillum " +  LINE_BREAK
                + "     dolore eu " +  LINE_BREAK
                + "     fugiat " +  LINE_BREAK
                + "     nulla " +  LINE_BREAK
                + "     pariatur. " +  LINE_BREAK
                + "     Excepteur " +  LINE_BREAK
                + "     sint " +  LINE_BREAK
                + "     obcaecat " +  LINE_BREAK
                + "     cupiditat " +  LINE_BREAK
                + "     non " +  LINE_BREAK
                + "     proident, " +  LINE_BREAK
                + "     sunt in " +  LINE_BREAK
                + "     culpa qui " +  LINE_BREAK
                + "     officia " +  LINE_BREAK
                + "     deserunt " +  LINE_BREAK
                + "     mollit " +  LINE_BREAK
                + "     anim id " +  LINE_BREAK
                + "     est " +  LINE_BREAK
                + "     laborum.";
        int numberOfFrontSpaces = 5;
        int lineLength = 15;
        TextBreaker breaker = new TextBreaker(lineLength, numberOfFrontSpaces);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void agBreaks() {
        String text = "Entfernt: [Amtsgericht]. HR-Ort: [München]. HR-Ort eindeutig. HR-Typ: "
                + "[HRB]. Nummer: [120704]. Zusatz: [].";
        String expected = "Entfernt: [Amtsgericht]. HR-Ort: [München]. HR-Ort eindeutig. "
                + "HR-Typ: [HRB]."
                + LINE_BREAK
                + "                Nummer: [120704]. Zusatz: [].";
        int lineLength = 80;
        int numberOfFrontSpaces = 16;
        TextBreaker breaker = new TextBreaker(lineLength, numberOfFrontSpaces);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void agBreaksKeepSpacesAtBreaks() {
        String text = "Entfernt: [Amtsgericht]. HR-Ort: [München]. HR-Ort eindeutig. HR-Typ: "
                + "[HRB]. Nummer: [120704]. Zusatz: [].";
        String expected = "Entfernt: [Amtsgericht]. HR-Ort: [München]. HR-Ort eindeutig. "
                + "HR-Typ: [HRB]. "
                + LINE_BREAK
                + "                Nummer: [120704]. Zusatz: [].";
        int lineLength = 80;
        int numberOfFrontSpaces = 16;
        TextBreaker breaker = new TextBreaker(lineLength, numberOfFrontSpaces);
        breaker.keepSpacesAtBreaks();
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

    @Test
    public void doNotBreakTooEarlyOnFirstLinesWhenAddSpaceToOterLines() {
        String text = "    public void updateEditFieldsByUser(String street, String co, "
                + "String postCode, String town, String phone, String fax, String email, "
                + "String internet) {";
        String expected = "    public void updateEditFieldsByUser(String street, String co, "
                + "String postCode, String town,"
                + LINE_BREAK
                + "            String phone, String fax, String email, "
                + "String internet) {";
        int lineLength = 100;
        int numberOfFrontSpaces = 12;
        TextBreaker breaker = new TextBreaker(lineLength, numberOfFrontSpaces);
        String actual = breaker.addLineBreaks(text);
        assertEquals(expected, actual);
    }

}

