package de.duehl.swing.ui.components.elements;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionListener;
import java.awt.event.KeyListener;

import javax.swing.JPanel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import de.duehl.swing.ui.colors.Colorizer;
import de.duehl.swing.ui.components.selections.OptionSelection;

/**
 * Diese Klasse stellt ein an- oder abwählbares Element mit einer Überschrift, einem frei
 * bearbeitbaren Text und einem Schalter, um diesen zu ändern, etwa in einem Optionendialog dar.
 *
 * @version 1.01     2017-03-01
 * @author Christian Dühl
 */

public class TextFieldWithChangeButtonAndOption {

    /** Überschrift. */
    private final OptionSelection titleBox;

    /** Textfeld und Schalter zur Änderung. */
    private final TextFieldWithChangeButton fieldWithButton;

    /** Panel mit Überschrift, Pfad und Änderungsbutton. */
    private final Component panel;

    /**
     * Konstruktor.
     *
     * @param title
     *            Überschrift für das Element.
     */
    public TextFieldWithChangeButtonAndOption(String title) {
        titleBox = new OptionSelection(title);
        fieldWithButton = new TextFieldWithChangeButton();
        addChangeListener(createCheckBoxChangeListener());
        panel = createPanel();
    }

    private ChangeListener createCheckBoxChangeListener() {
        return new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                reactOnCheckBoxClick();
            }

        };
    }

    private void reactOnCheckBoxClick() {
        boolean selected = titleBox.isSelected();
        fieldWithButton.setEnabled(selected);
        // if (selected) { fieldWithButton.setEditFieldColors(); }
    }

    /** Fügt einen ChangeListener zur CheckBox hinzu. */
    private void addChangeListener(ChangeListener changeListener) {
        titleBox.addChangeListener(changeListener);
    }

    /** Fragt ab, ob die Option ausgewählt ist. */
    public boolean isSelected() {
        return titleBox.isSelected();
    }

    /** Setzt die Option auf ausgewählt oder nicht ausgewählt. */
    public void setSelected(boolean selected) {
        titleBox.setSelected(selected);
    }

    /** Erzeugt den Panel mit Überschrift, Option und Änderungsbutton. */
    private Component createPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(titleBox.getCheckBox(), BorderLayout.WEST);
        panel.add(fieldWithButton.getPanel(), BorderLayout.CENTER);

        return panel;
    }

    /** Getter für das Panel mit Überschrift, Pfad und Änderungsbutton. */
    public Component getPanel() {
        return panel;
    }

    /** Färbt alle Komponenten mit dem übergebenen Colorizer ein. */
    public void colorize(Colorizer colorizer) {
        colorizer.setColors(panel);
        titleBox.colorize(colorizer);
        fieldWithButton.colorize(colorizer);
    }

    /** Fügt einen ActionListener zum "ändern"-Schalter hinzu. */
    public void addButtonActionListener(ActionListener actionListener) {
        fieldWithButton.addButtonActionListener(actionListener);
    }

    /** Getter für den (getrimmten) Text des Textfeldes. */
    public String getText() {
        return fieldWithButton.getText();
    }

    /** Setzt den Text des Textfeldes. */
    public void setText(String text) {
        fieldWithButton.setText(text);
    }

    /** Setter für den auf dem Änderungsbutton angezeigten Text. */
    public void setButtonText(String title) {
        fieldWithButton.setButtonText(title);
    }

    /** Setter für die gewünschte Größe des Textfeldes. */
    public void setPreferredSize(Dimension preferredSize) {
        fieldWithButton.setPreferredSize(preferredSize);
    }

    /** Setzt die bevorzugt Größe des Buttons. */
    public void setButtonPreferredSize(Dimension buttonDimension) {
        fieldWithButton.setButtonPreferredSize(buttonDimension);
    }

//    /** Aktiviert oder deaktiviert Textfeld und Button. */
//    public void setEnabled(boolean selected) {
//        fieldWithButton.setEnabled(selected);
//    }

    /** Löst ein Anklicken des Buttons aus. */
    public void clickButton() {
        fieldWithButton.clickButton();
    }

    /** Fügt einen KeyListener zum Textfeld hinzu. */
    public void addKeyListener(KeyListener keyListener) {
        fieldWithButton.addKeyListener(keyListener);
    }

    /** Fügt einen KeyListener zum Textfeld hinzu, der bei Return den Button auslöst. */
    public void addReturnKeyClickButtonListener() {
        fieldWithButton.addReturnKeyClickButtonListener();
    }

}
