package de.duehl.swing.ui.components.selections;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.FocusListener;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextArea;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.Colorizer;

/**
 * Diese Klasse stellt die Auswahl eines Textes etwa in einem Optionendialog dar.
 *
 * @version 1.01     2017-03-30
 * @author Christian Dühl
 */

public class LongStringSelection {

    /** Überschrift für die Wahl dieser Datei oder dieses Verzeichnisses. */
    private final JLabel titleLabel;

    /** Panel mit Überschrift und Eingabefeld. */
    private final Component mainPanel;

    /** Eingabebereich für den Text. */
    private final JTextArea textArea;

    /**
     * Konstruktor.
     *
     * @param title
     *            Überschrift für die Wahl dieser Datei oder dieses Verzeichnisses.
     */
    public LongStringSelection(String title) {
        titleLabel = new JLabel(title);

        textArea = new JTextArea();
        GuiTools.setEditAreaColors(textArea);
        textArea.setWrapStyleWord(true);
        textArea.setLineWrap(true);

        mainPanel = createPanel();
    }

    /** Erstellt den Panel mit der Überschrift und dem Eingabefeld. */
    private Component createPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(titleLabel, BorderLayout.NORTH);
        panel.add(createTextArea(), BorderLayout.CENTER);

        return panel;
    }

    /** Erstellt das Eingabefeld. */
    private Component createTextArea() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        //panel.add(new JScrollPane(textArea), BorderLayout.CENTER);
        panel.add(textArea, BorderLayout.CENTER);

        return panel;
    }

    /** Getter für die (getrimmte) Option. */
    public String getText() {
        String text = textArea.getText();
        text = text.trim();
        return text;
    }

    /** Setter für die Option. */
    public void setText(String text) {
        textArea.setText(text);
    }

    /** Getter für das Panel mit Überschrift und Eingabefeld. */
    public Component getPanel() {
        return mainPanel;
    }

    /** Färbt alle Komponenten mit dem übergebenen Colorizer ein. */
    public void colorize(Colorizer colorizer) {
        colorizer.setColors(mainPanel);
        colorizer.setColors(titleLabel);
        colorizer.setColors(textArea);
        colorizer.setEditAreaColors(textArea);
    }

    /** Setter für die gewünschte Größe des Textfeldes. */
    public void setPreferredSize(Dimension preferredSize) {
        textArea.setPreferredSize(preferredSize);
    }

    /** Fügt einen FocusListener zum Textfeld hinzu. */
    public void addTextFieldFocuslistener(FocusListener focusListener) {
        textArea.addFocusListener(focusListener);
    }

    /** Legt fest, ob das Textfeld editierbar ist. */
    public void setEditable(boolean editable) {
        textArea.setEditable(editable);
    }

}
