package de.duehl.swing.ui.elements.search;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.elements.textfield.EnterTextField;

/**
 * Diese Klasse stellt ein Eingabefeld und zwei Buttons zum Suchen und Weitersuchen zur Verfügung.
 *
 * @version 1.01     2017-12-15
 * @author Christian Dühl
 */

public class SearchPanel extends JPanel {

    private static final long serialVersionUID = 1L;

    /** Reagiert auf die Eingaben in diesem SearchPanel. */
    private final SearchReaktor searchReaktor;

    /** Das Feld mit dem zu suchenden Text. */
    private final EnterTextField searchTextField;

    /**
     * Konstruktor.
     *
     * @param searchReaktor
     *            Reagiert auf die Eingaben in diesem SearchPanel.
     */
    public SearchPanel(SearchReaktor searchReaktor) {
        setLayout(new BorderLayout());
        this.searchReaktor = searchReaktor;

        searchTextField = new EnterTextField(text -> search(text));
        GuiTools.biggerFont(searchTextField, 2);

        add(searchTextField, BorderLayout.CENTER);
        add(createSearchButtons(), BorderLayout.EAST);
    }

    private Component createSearchButtons() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createSearchButton(), BorderLayout.NORTH);
        panel.add(createSearchAgainButton(), BorderLayout.SOUTH);

        return panel;
    }

    private Component createSearchButton() {
        JButton button = createButton("Suchen");
        button.addActionListener(e -> searchButtonPressed());
        return button;
    }

    private JButton createButton(String text) {
        JButton button = new JButton(text);
        button.setBorder(BorderFactory.createRaisedBevelBorder());
        //button.setBorder(BorderFactory.createEmptyBorder());
        button.setOpaque(true);
        button.setBackground(new Color(230, 230, 255));
        GuiTools.smalerFont(button, 2);
        return button;
    }

    private void searchButtonPressed() {
        search(getTrimmedSearchText());
    }

    private String getTrimmedSearchText() {
        return searchTextField.getText().trim();
    }

    private Component createSearchAgainButton() {
        JButton button = createButton("Weitersuchen");
        button.addActionListener(e -> searchAgainButtonPressed());
        return button;
    }

    private void searchAgainButtonPressed() {
        searchAgain();
    }

    private void search(String text) {
        searchReaktor.search(text);
    }

    private void searchAgain() {
        searchReaktor.searchAgain();
    }

}

/*
TODO: SearchPanelTest und SearchPanelDemo mit richtigem Text und der Suchlogik und Weitersuchlogik!
*/
