package de.duehl.swing.ui.elements.watch;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.datetime.time.watch.CountDownWatch;

/**
 * Diese Klasse stellt ein Label dar, das den auf ihm angezeigten Countdown automatisch
 * aktualisiert.
 *
 * @version 1.01     2021-03-17
 * @author Christian Dühl
 */

public class CountDownWatchLabel extends WatchBaseLabel {

    private static final long serialVersionUID = 1L;

    /** Countdown, dessen Zeit angezeigt wird. */
    private final CountDownWatch countDownWatch;

    /**
     * Konstruktor ohne Text vor und nach dem Countdown.
     *
     * @param duration
     *            Dauer des Countdowns in der Form HH:MM:SS.
     */
    public CountDownWatchLabel(String duration) {
        this("", new CountDownWatch(duration), "");
    }

    /**
     * Konstruktor mit Text vor dem Countdown.
     *
     * @param duration
     *            Dauer des Countdowns in der Form HH:MM:SS.
     * @param startText
     *            Text vor dem Countdown.
     */
    public CountDownWatchLabel(String duration, String startText) {
        this(startText, new CountDownWatch(duration), "");
    }

    /**
     * Konstruktor mit Text vor und nach dem Countdown.
     *
     * @param duration
     *            Dauer des Countdowns in der Form HH:MM:SS.
     * @param startText
     *            Text vor dem Countdown.
     * @param endText
     *            Text nach dem Countdown.
     */
    public CountDownWatchLabel(String duration, String startText, String endText) {
        this(startText, new CountDownWatch(duration), endText);
    }

    /**
     * Konstruktor ohne Text vor und nach dem Countdown.
     *
     * @param duration
     *            Dauer des Countdowns in Sekunden.
     */
    public CountDownWatchLabel(long duration) {
        this("", new CountDownWatch(duration), "");
    }

    /**
     * Konstruktor mit Text vor dem Countdown.
     *
     * @param duration
     *            Dauer des Countdowns in Sekunden.
     * @param startText
     *            Text vor dem Countdown.
     */
    public CountDownWatchLabel(long duration, String startText) {
        this(startText, new CountDownWatch(duration), "");
    }

    /**
     * Konstruktor mit Text vor und nach dem Countdown.
     *
     * @param duration
     *            Dauer des Countdowns in Sekunden.
     * @param startText
     *            Text vor dem Countdown.
     * @param endText
     *            Text nach dem Countdown.
     */
    public CountDownWatchLabel(long duration, String startText, String endText) {
        this(startText, new CountDownWatch(duration), endText);
    }

    /**
     * Konstruktor mit Text vor und nach dem Countdown sowie eigener Countdown-Uhr, die übergeben
     * wird.
     *
     * @param startText
     *            Text vor dem Countdown.
     * @param countDownWatch
     *            Countdown, dessen Zeit angezeigt wird.
     * @param endText
     *            Text nach dem Countdown.
     */
    public CountDownWatchLabel(String startText, CountDownWatch countDownWatch, String endText) {
        super(startText, countDownWatch, endText);
        this.countDownWatch = countDownWatch;
    }

    /** Startet den Countdown. */
    public void start() {
        stopActualisationRunnable();

        countDownWatch.start();

        createRunnableAndStartActualisationThread();
    }

    /** Setter für die Laufzeit des Countdowns ins Millisekunden. */
    public void setDurationInSeconds(long durationSeconds) {
        countDownWatch.setDurationInSeconds(durationSeconds);
    }

    /** Stringrepräsentation. */
    @Override
    public String toString() {
        return "CountDownLabel [" + createWatchLabelText() + "]";
    }

}
