package de.duehl.swing.ui.tables;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Component;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JTable;

import de.duehl.swing.ui.GuiTools;

/**
 * Diese Klasse ist für das Rendern (die optische Darstellung) der Tabellenzellen zuständig. Sie
 * erweitert den DifferentBackgroundsTableRenderer, welcher für abwechselnde Hintergrundfarben
 * sorgt, um Schriftgröße und die Möglichkeit, einige Spalten rechtsbündig zu gestalten.
 *
 * @version 1.01     2020-06-29
 * @author Christian Dühl
 */

public class SizableAlignableDifferentBackgroundsTableRenderer
        extends DifferentBackgroundsTableRenderer {

    private static final long serialVersionUID = 1L;

    /** Schriftgröße der Tabelle. */
    private final int fontSize;

    /** Liste der rechtsbündig zu formatierenden Spalten-Indices. */
    private List<Integer> rightAlignedColumnIndices;

    /**
     * Konstruktor.
     *
     * @param foregroundColor
     *            Vordergrundfarbe der Zellen.
     * @param backgroundColor
     *            Hintergrundfarbe der Zellen.
     * @param fontSize
     *            Schriftgröße der Tabelle.
     */
    public SizableAlignableDifferentBackgroundsTableRenderer(Color foregroundColor,
            Color backgroundColor, int fontSize) {
        super(foregroundColor, backgroundColor);
        this.fontSize = fontSize;

        rightAlignedColumnIndices = new ArrayList<>();
    }

    /** Setter für die Liste der rechtsbündig zu formatierenden Spalten-Indices. */
    public void setRightAlignedColumnIndices(List<Integer> rightAlignedColumnIndices) {
        this.rightAlignedColumnIndices = rightAlignedColumnIndices;
    }

    /**
     * Liefert eine Komponente für die anzuzeigende Zelle zurück. Die Hauptarbeit lassen wir
     * super.getTableCellRendererComponent(...) machen und hängen nur ein wenig Code dahinter.
     *
     * @param table
     *            Die Tabelle
     * @param value
     *            Das anzuzeigende Objekt (im Normalfall ein String)
     * @param isSelected
     *            Zeigt an, ob das Objekt selektiert wurde.
     * @param hasFocus
     *            Zeigt an, ob der Focus auf dieser Zelle liegt.
     * @param row
     *            Zeile in der die Zelle liegt.
     * @param column
     *            Spalte in der die Zelle liegt.
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
            boolean hasFocus, int row, int column) {
        JLabel createdLabel = (JLabel) super.getTableCellRendererComponent(table, value, isSelected,
                hasFocus, row, column);
        GuiTools.setFontSize(createdLabel, fontSize);
        if (rightAlignedColumnIndices.contains(column)) {
            createdLabel.setHorizontalAlignment(JLabel.RIGHT);
        }
        return createdLabel;
    }

}
