package de.duehl.swing.ui.undoredo;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Component;
import java.awt.GridLayout;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.basics.history.HistoryStateDisplayer;
import de.duehl.basics.history.HistoryUser;
import de.duehl.swing.ui.resources.IconLoader;
import de.duehl.swing.ui.undoredo.resources.IconDefinitions;

/**
 * Diese Klasse stellt ein Element mit zwei Buttons für undo und redo dar.
 *
 * @version 1.01     2017-12-19
 * @author Christian Dühl
 */

public class UndoRedoBar implements HistoryStateDisplayer {

    private final HistoryUser historyUser;
    private final IconLoader iconLoader;
    private final JPanel panel;

    private JButton undoButton;
    private JButton redoButton;

    public UndoRedoBar(HistoryUser historyUser) {
        this.historyUser = historyUser;

        IconDefinitions iconDefinitions = new IconDefinitions();
        iconLoader = iconDefinitions.createIconLoader();

        panel = new JPanel();

        populatePanel();
    }

    private void populatePanel() {
        panel.setLayout(new GridLayout(1, 0, 2, 2));

        panel.add(createUndoButton());
        panel.add(createRedoButton());
    }

    private Component createUndoButton() {
        undoButton = createButton(IconDefinitions.HISTORY_PREVIOUS);
        undoButton.addActionListener(e -> historyUser.undo());
        undoButton.setToolTipText("Rückgängig");
        return undoButton;
    }

    private Component createRedoButton() {
        redoButton = createButton(IconDefinitions.HISTORY_NEXT);
        redoButton.addActionListener(e -> historyUser.redo());
        redoButton.setToolTipText("Wiederherstellen");
        return redoButton;
    }

    private JButton createButton(String iconName) {
        JButton button = new JButton(iconLoader.loadIcon(iconName));
        button.setBorder(BorderFactory.createRaisedBevelBorder());
        button.setOpaque(true);
        return button;
    }

    /**
     * Lässt die Benutzung z.B. des Buttons und des Menüeintrags für den vorigen Stand aus der
     * Bearbeitungshistorie zu.
     */
    @Override
    public void enableUndo() {
        undoButton.setEnabled(true);
    }

    /**
     * Verbietet die Benutzung z.B. des Buttons und des Menüeintrags für den vorigen Stand aus der
     * Bearbeitungshistorie.
     */
    @Override
    public void disableUndo() {
        undoButton.setEnabled(false);
    }

    /**
     * Lässt die Benutzung z.B. des Buttons und des Menüeintrags für den nächsten Stand aus der
     * Bearbeitungshistorie zu.
     */
    @Override
    public void enableRedo() {
        redoButton.setEnabled(true);
    }

    /**
     * Verbietet die Benutzung z.B. des Buttons und des Menüeintrags für den nächsten Stand aus der
     * Bearbeitungshistorie.
     */
    @Override
    public void disableRedo() {
        redoButton.setEnabled(false);
    }

    /** Gibt die Komponente der Button-Bar zurück, um sie in die Gui einzufügen. */
    public Component getComponent() {
        return panel;
    }

}
