package de.duehl.swing.text.html;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

public class TestText2HtmlTest {

    private static final String HTML_HEAD = "<html><head></head><body><span style=\"font-size:20pt\">";
    private static final String HTML_FEET = "</span></body></html>";
    // TODO div statt span? In der Anzeige im Programm anschauen ... vielleicht macht das Leerraum

    @Test
    public void colorTextWithEmptyText() {
        String text = "";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithoutKeywords() {
        String text = "Im Wald steht ein Baum.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD + text + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithFruit() {
        String text = "Im Wald liegt eine Birne.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Im Wald liegt eine "
                + "<span style=\"color:ff0000;font-weight:bold\">Birne</span>."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithFruitsAndExpandWord() {
        String text = "Im Wald wachsen Blaubeeren.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Im Wald wachsen "
                + "<span style=\"color:ff0000;font-weight:bold\">Blaubeeren</span>."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithFruitsAndExpandWord2() {
        String text = "Im Wald steht ein Apfelbaum.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Im Wald steht ein "
                + "<span style=\"color:ff0000;font-weight:bold\">Apfelbaum</span>."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithVegetable() {
        String text = "Im Garten wächst eine Gurke.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Im Garten wächst eine "
                + "<span style=\"color:006400;font-style:italic\">Gurke</span>."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithChessPiece() {
        String text = "Der König zieht nach f3.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Der "
                + "<span style=\"color:db7093;background-color:ffc8c8;font-weight:bold\">König</span>"
                + " zieht nach f3."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithChessPiece2() {
        String text = "Der Läufer zieht von d2 nach h6.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Der "
                + "<span style=\"color:db7093;background-color:ffc8c8;font-weight:bold\">Läufer</span>"
                + " zieht von d2 nach h6."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithChessPhraseButPieceWins() {
        String text = "Schwarzer Angriff auf dem Königsflügel!";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Schwarzer Angriff auf dem "
                + "<span style=\"color:db7093;background-color:ffc8c8;font-weight:bold\">"
                + "Königsflügel</span>"
                + "!"
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithChessPhrase() {
        String text = "Abzugsschach auf d8!";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "<span style=\"color:ff1493;background-color:ffffe0;font-weight:bold\">"
                + "Abzugsschach</span>"
                + " auf d8!"
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithFuzziness() { // in Text2Html ist Deciusstraße hinterlegt!
        String text = "Adresse: Deciusstr. 6, 33611 Bielefeld.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Adresse: "
                + "<span style=\"color:8a2be2;font-weight:bold\">"
                + "Deciusstr.</span>"
                + " 6, 33611 Bielefeld."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithSentences() {
        String text = "Die Sonne scheint. "
                + "Der Apfel hängt am Birnbaum. "
                + "Dort steht ein Haus. "
                + "Der Läufer zieht von c1 nach g5. "
                + "Morgen wird es regnen.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Die Sonne scheint. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der Apfel hängt am Birnbaum.</span> "
                + "Dort steht ein Haus. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der Läufer zieht von c1 nach g5.</span> "
                + "Morgen wird es regnen."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithSentencesAllowNestedSpans() {
        String text = "Die Sonne scheint. "
                + "Der Apfel hängt am Birnbaum. "
                + "Dort steht ein Haus. "
                + "Der Läufer zieht von c1 nach g5. "
                + "Morgen wird es regnen.";
        TestText2Html text2Html = new TestText2Html();
        text2Html.allowNestedSpans();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Die Sonne scheint. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der "
                + "<span style=\"color:ff0000;font-weight:bold\">Apfel</span> "
                + "hängt am Birnbaum.</span> "
                + "Dort steht ein Haus. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der "
                + "<span style=\"color:db7093;background-color:ffc8c8;font-weight:bold\">Läufer"
                + "</span> "
                + "zieht von c1 nach g5.</span> "
                + "Morgen wird es regnen."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithSentencesHideOtherHighlightsInSentences() {
        String text = "Die Sonne scheint. "
                + "Der Apfel hängt am Birnbaum. "
                + "Dort steht ein Haus. "
                + "Der Läufer zieht von c1 nach g5. "
                + "Morgen wird es regnen.";
        TestText2Html text2Html = new TestText2Html();
        text2Html.hideOtherHighlightsInSentences();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Die Sonne scheint. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der Apfel hängt am Birnbaum.</span> "
                + "Dort steht ein Haus. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der Läufer zieht von c1 nach g5.</span> "
                + "Morgen wird es regnen."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithSentencesAllowNestedSpansAndHideOtherHighlightsInSentences() {
        String text = "Die Sonne scheint. "
                + "Der Apfel hängt am Birnbaum. "
                + "Dort steht ein Haus. "
                + "Der Läufer zieht von c1 nach g5. "
                + "Morgen wird es regnen.";
        TestText2Html text2Html = new TestText2Html();
        text2Html.allowNestedSpans();
        text2Html.hideOtherHighlightsInSentences();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Die Sonne scheint. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der Apfel hängt am Birnbaum.</span> "
                + "Dort steht ein Haus. "
                + "<span style=\"color:eee8aa;font-weight:bold\">"
                + "Der Läufer zieht von c1 nach g5.</span> "
                + "Morgen wird es regnen."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

    @Test
    public void colorTextWithAll() {
        String text = ""
                + "Im Wald steht ein Baum. "
                + "Im Wald liegt eine Birne. "
                + "Im Wald wachsen Blaubeeren. "
                + "Im Wald steht ein Apfelbaum. "
                + "Im Garten wächst eine Gurke. "
                + "Der König zieht nach f3. "
                + "Schwarzer Angriff auf dem Königsflügel! "
                + "Abzugsschach auf d8! "
                + "Adresse: Deciusstr. 6, 33611 Bielefeld.";
        TestText2Html text2Html = new TestText2Html();
        String actual = text2Html.colorText(text);
        String expected = HTML_HEAD
                + "Im Wald steht ein Baum. "

                + "Im Wald liegt eine "
                + "<span style=\"color:ff0000;font-weight:bold\">Birne</span>. "

                + "Im Wald wachsen "
                + "<span style=\"color:ff0000;font-weight:bold\">Blaubeeren</span>. "

                + "Im Wald steht ein "
                + "<span style=\"color:ff0000;font-weight:bold\">Apfelbaum</span>. "

                + "Im Garten wächst eine "
                + "<span style=\"color:006400;font-style:italic\">Gurke</span>. "

                + "Der "
                + "<span style=\"color:db7093;background-color:ffc8c8;font-weight:bold\">König</span>"
                + " zieht nach f3. "

                + "Schwarzer Angriff auf dem "
                + "<span style=\"color:db7093;background-color:ffc8c8;font-weight:bold\">"
                + "Königsflügel</span>"
                + "! "

                + "<span style=\"color:ff1493;background-color:ffffe0;font-weight:bold\">"
                + "Abzugsschach</span>"
                + " auf d8! "

                + "Adresse: "
                + "<span style=\"color:8a2be2;font-weight:bold\">"
                + "Deciusstr.</span>"
                + " 6, 33611 Bielefeld."
                + HTML_FEET;
        assertEquals(expected, actual);
    }

}
