package de.duehl.vocabulary.japanese.grammar;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.vocabulary.japanese.data.Vocable;

/**
 * Diese Klasse hält eine sortierte Liste der Adjektivformen vor und kann eine Liste von Vokabeln
 * nach diesen sortieren.
 *
 * @version 1.01     2025-11-26
 * @author Christian Dühl
 */

public class AdjectiveSearchWords {

    public static final String I_ADJECTIVE_POSITIVE_PRESENCE = "Positive Gegenwart I-Adjektiv";
    public static final String NA_ADJECTIVE_POSITIVE_PRESENCE = "Positive Gegenwart Na-Adjektiv";

    /** Die sortierte Liste der Adjektivformen von I-Adjektiven. */
    public static final List<String> SORTED_I_ADJEKTIVE_SEARCH_WORDS =
            CollectionsHelper.buildListFrom(
                    I_ADJECTIVE_POSITIVE_PRESENCE,
                    "Negative Gegenwart I-Adjektiv",
                    "Positive Vergangenheit I-Adjektiv",
                    "Negative Vergangenheit I-Adjektiv"
                    );

    /** Die sortierte Liste der Adjektivformen von Na-Adjektiven. */
    public static final List<String> SORTED_NA_ADJEKTIVE_SEARCH_WORDS =
            CollectionsHelper.buildListFrom(
                    NA_ADJECTIVE_POSITIVE_PRESENCE,
                    "Negative Gegenwart Na-Adjektiv",
                    /*
                     * Die positive Vergangenheit wird nicht am Na-Adjektiv konjugiert, sondern
                      aus desu wird deshita.
                     */
                    "Negative Vergangenheit Na-Adjektiv"
                    );

    /** Die sortierte Liste der Adjektivformen. */
    private static final List<String> SORTED_ADJEKTIVE_SEARCH_WORDS =
            CollectionsHelper.joinLists(SORTED_I_ADJEKTIVE_SEARCH_WORDS,
                    SORTED_NA_ADJEKTIVE_SEARCH_WORDS);

    /** Sortiert die übergebene Liste von Vokabeln nach den Suchbegriffen für Adjektivformen. */
    public static void sortVocablesByAdjectiveForms(List<Vocable> vocables) {
        Collections.sort(vocables, new Comparator<Vocable>() {
            @Override
            public int compare(Vocable v1, Vocable v2) {
                int index1 = determineLowestExistingAdjectiveFormIndex(v1.getSearchWords());
                int index2 = determineLowestExistingAdjectiveFormIndex(v2.getSearchWords());
                if (index1 == -1) {
                    index1 = Integer.MAX_VALUE;
                }
                if (index2 == -1) {
                    index2 = Integer.MAX_VALUE;
                }
                return index1 - index2;
            }});
    }

    /**
     * Bestimmt den kleinsten Index der vorkommenden Verbformen unter den Suchbegriffen. Wird keine
     * Adjektivformen darin gefunden, so wird die größte Integerzahl zurückgegeben.
     */
    private static int determineLowestExistingAdjectiveFormIndex(List<String> searchWords) {
        int lowestIndex = Integer.MAX_VALUE;

        for (String searchWord : searchWords) {
            int index = SORTED_ADJEKTIVE_SEARCH_WORDS.indexOf(searchWord);
            if (index != -1 && index < lowestIndex) {
                lowestIndex = index;
            }
        }

        return lowestIndex;
    }

}
