package de.duehl.vocabulary.japanese.ui.components;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;

import javax.swing.JPanel;
import javax.swing.JTextPane;

import de.duehl.basics.io.FileHelper;
import de.duehl.swing.ui.buttons.painted.GoButton;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalVocableData;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.tools.VocabularyTools;
import de.duehl.vocabulary.japanese.ui.components.display.KanjiAndKanaDisplay;
import de.duehl.vocabulary.japanese.ui.components.display.RomajiAndPronunciationDisplay;
import de.duehl.vocabulary.japanese.ui.components.display.TranslationCommentAndVocabularyDescriptionDisplay;

/**
 * Diese Klasse zeigt eine Vokabel an. Dies ist die sogenannte Einzeldarstellung der Vokabeln.
 *
 * @version 1.01     2025-11-15
 * @author Christian Dühl
 */

public class VocableViewer {

    /** Die Programmoptionen. */
    private final Options options;

    /** Der Panel auf dem die Vokabel dargestellt wird. */
    private final JPanel panel;

    /** Die TextPane in der die Kanji und Kana dargestellt werden. */
    private final JTextPane kanjiAndKanaPane;

    /** Die TextPane in der die Romaji und Aussprache dargestellt werden. */
    private final JTextPane romajiAndPronunciationPane;

    /**
     * Die TextPane in der die Übersetzungen, den Kommentar und die Beschreibung des Vokabulars
     * dargestellt werden.
     */
    private final JTextPane translationCommentAndVocabularyDescriptionPane;

    private final GoButton playMp3Button;

    /** Die Vokabel, welche angezeigt wird. */
    private Vocable vocable;

    /**
     * Gibt an, ob trotz der Einstellung in den Optionen alle Übersetzungen angezeigt werden
     * sollen.
     *
     * Dies wird nur im Dialog mit den Details der Vokabel verwendet, um auch oben im bunten Teil
     * der Vokabeldarstellung alle Übersetzungen anzuzeigen.
     */
    private boolean showAllTranslationsForUseInDetailsDialog;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     */
    public VocableViewer(Options options) {
        this.options = options;

        panel = new JPanel();
        kanjiAndKanaPane = new JTextPane();
        romajiAndPronunciationPane = new JTextPane();
        translationCommentAndVocabularyDescriptionPane = new JTextPane();
        playMp3Button = new GoButton();

        showAllTranslationsForUseInDetailsDialog = false;

        initSelections();
        createPanel();
    }

    private void initSelections() {
        playMp3Button.addActionListener(e -> playMp3());
        playMp3Button.setForceSquare(true);
        playMp3Button.setPreferredSize(new Dimension(50, 50));
    }

    private void createPanel() {
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(createKanjiKanaAndMp3PlayButtonArea());
        panel.add(romajiAndPronunciationPane);
        panel.add(translationCommentAndVocabularyDescriptionPane);
    }

    private Component createKanjiKanaAndMp3PlayButtonArea() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(kanjiAndKanaPane, BorderLayout.CENTER);
        panel.add(playMp3Button, BorderLayout.EAST);

        return panel;
    }

    /**
     * Legt fest, dass trotz der Einstellung in den Optionen alle Übersetzungen angezeigt werden
     * sollen.
     *
     * Dies wird nur im Dialog mit den Details der Vokabel verwendet, um auch oben im bunten Teil
     * der Vokabeldarstellung alle Übersetzungen anzuzeigen.
     */
    public void showAllTranslationsForUseInDetailsDialog() {
        showAllTranslationsForUseInDetailsDialog = true;
    }

    /** Zeigt eine neue Variable an. */
    public void showVocable(Vocable vocable, InternalAdditionalVocableData data) {
        this.vocable = vocable;

        VocableColors vocableColors = new VocableColors(options);
        Color backgroundColor = vocableColors.determineBackgroundColor(data,
                options.isColorVocableDependingOnLastSuccessWhenViewingVocable(),
                options.getTranslationDirection());

        KanjiAndKanaDisplay display1 = new KanjiAndKanaDisplay(vocable, kanjiAndKanaPane,
                backgroundColor, options);
        display1.display();

        playMp3Button.setBackgroundColor(backgroundColor);

        RomajiAndPronunciationDisplay display2 = new RomajiAndPronunciationDisplay(vocable,
                romajiAndPronunciationPane, backgroundColor, options);
        display2.display();

        TranslationCommentAndVocabularyDescriptionDisplay display3 =
                new TranslationCommentAndVocabularyDescriptionDisplay(vocable,
                        translationCommentAndVocabularyDescriptionPane, backgroundColor, options);
        if (!options.isShowAllTranslations() && !showAllTranslationsForUseInDetailsDialog) {
            display3.setNumberOfShownTranslations(options.getNumberOfShownTranslations());
        }
        display3.display();

        String mp3 = vocable.getMp3();
        playMp3Button.setEnabled(!mp3.isBlank() && FileHelper.isFile(mp3));

        panel.repaint();
    }

    /** Spielt die zugehörige MP3 ab. */
    public void playMp3() {
        String mp3 = vocable.getMp3();
        VocabularyTools.playMp3(mp3);
    }

    /** Getter für den Panel auf dem die Vokabel dargestellt wird. */
    public Component getPanel() {
        return panel;
    }

}
