package de.duehl.vocabulary.japanese.ui.dialog.kanji.kanjitest;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.util.List;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.components.selections.SelectionsHelper;
import de.duehl.swing.ui.components.selections.StringSelection;
import de.duehl.swing.ui.components.selections.TextAreaSelection;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;
import de.duehl.swing.ui.key.BindKeysOnRootPane;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.common.color.VocableColors;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.test.data.KanjiUserInputType;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.test.data.SingleUserInputKanjiCheckResult;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.dialog.kanji.display.KanjiPanel;

/**
 * Diese Klasse zeigt den Erfolg oder Misserfolg nach dem Abfragen eines Kanji an.
 *
 * @version 1.01     2025-11-21
 * @author Christian Dühl
 */

public class KanjiTestEvaluationDialog extends ModalDialogBase {

    /** Minimale Breite des Fensters, ansonsten passt es sich an den Inhalt an. */
    private static final int MIN_WIDTH = 700;


    /** Das Ergebnis der Prüfung. */
    private final SingleUserInputKanjiCheckResult result;

    /** Das Kanji welches abgefragt wurde. */
    private final Kanji kanji;

    /** Gibt an, ob die Übersetzung korrekt war. */
    private final boolean correct;

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Die anzuzeigende Nachricht. */
    private final String message;

    /** Das Objekt das passende Farben erzeugt. */
    private final VocableColors vocableColors;

    /** Die Hintergrundfarbe. */
    private final Color backgroundColor;

    /** Das UI-Komponente, auf der die Daten des Kanji dargestellt werden. */
    private final KanjiPanel kanjiPanel;

    /** Der Button zum Beenden des Dialogs. */
    private final JButton quitButton;

    /** Gibt an, ob im Fall einer falsch beantworteten Vokabel nur ein Tippfehler vorlag. */
    private boolean onlyTypingError;

    private final StringSelection userInputGermanMeaningSelection;
    private final StringSelection userInputOnLesungenSelection;
    private final StringSelection userInputKunLesungenSelection;

    private final TextAreaSelection errorMessageSelection;
    private final TextAreaSelection memorandumSelection;

    private boolean errorFilled;
    private boolean memorandumFilled;


    /**
     * Konstruktor.
     *
     * @param result
     *            Das Ergebnis der Bewertung der Benutzereingaben inklusive Kanji, welches
     *            abgefragt wurde.
     * @param userInputGermanMeaning
     *            Die vom Benutzer eingegebene deutsche Bedeutung.
     * @param userInputOnLesung
     *            Die vom Benutzer eingebenen ON-Lesungen.
     * @param userInputKunLesung
     *            Die vom Benutzer eingebenen kun-Lesungen.
     * @param correct
     *            Gibt an, ob die Abfrage korrekt war.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     */
    public KanjiTestEvaluationDialog(SingleUserInputKanjiCheckResult result,
            String userInputGermanMeaning, String userInputOnLesung, String userInputKunLesung,
            FumikoDataStructures dataStructures, FumikoUiObjects uiObjects, Point parentLocation) {
        super(parentLocation, uiObjects.getProgramImage(),
                result.isOk() ? "Richtig!" : "Leider falsch");
        addEscapeBehaviour();

        this.result = result;
        this.kanji = result.getKanji();
        this.correct = result.isOk();
        this.dataStructures = dataStructures;

        onlyTypingError = false;

        message = correct ? "Richtig beantwortet, sehr gut!" : "Das war leider nicht richtig";

        vocableColors = new VocableColors(dataStructures.getOptions());
        backgroundColor = correct
                ? vocableColors.getSuccessColor()
                : vocableColors.getFailureColor();

        quitButton = new JButton("Beenden");

        kanjiPanel = new KanjiPanel(kanji, getWindowAsComponent());
        userInputGermanMeaningSelection = new StringSelection("Eingebene deutsche Bedeutung");
        userInputOnLesungenSelection = new StringSelection("Eingebene ON-Lesungen");
        userInputKunLesungenSelection = new StringSelection("Eingebene kun-Lesungen");
        errorMessageSelection = new TextAreaSelection("Fehlermeldungen");
        memorandumSelection = new TextAreaSelection("Hinweise");

        init(userInputGermanMeaning, userInputOnLesung, userInputKunLesung);
        fillDialog();
    }

    private void init(String userInputGermanMeaning, String userInputOnLesung,
            String userInputKunLesung) {
        initKanjiPanel();
        fillKanjiPanel();

        initUserInputTextSelections();
        fillUserInputTextSelections(userInputGermanMeaning, userInputOnLesung, userInputKunLesung);

        initResultTextSelections();
        fillResultTextSelections();

        initQuitButton();
    }

    private void initKanjiPanel() {
        kanjiPanel.createGui();
    }

    private void fillKanjiPanel() {
        kanjiPanel.showKanji();
    }

    private void initUserInputTextSelections() {
        SelectionsHelper.initSelectionAsViewer(userInputGermanMeaningSelection);
        SelectionsHelper.initSelectionAsViewer(userInputOnLesungenSelection);
        SelectionsHelper.initSelectionAsViewer(userInputKunLesungenSelection);

        Options options = dataStructures.getOptions();
        if (options.isColorUserInputAtKanjiTestEvaluationDialog()) {
            setUserInputTextSelectionsBackgrounds();
        }
    }

    private void setUserInputTextSelectionsBackgrounds() {
        List<KanjiUserInputType> wrongKanjiUserInputTypes = result.getWrongKanjiUserInputTypes();
        boolean germanMeaningCorrect = !wrongKanjiUserInputTypes.contains(
                KanjiUserInputType.GERMAN_MEANING);
        boolean onLesungCorrect = !wrongKanjiUserInputTypes.contains(
                KanjiUserInputType.ON_LESUNG);
        boolean kunLesungCorrect = !wrongKanjiUserInputTypes.contains(
                KanjiUserInputType.KUN_LESUNG);

        Color germanMeaningColor = getSuccessOrFailureColor(germanMeaningCorrect);
        Color onLesungColor = getSuccessOrFailureColor(onLesungCorrect);
        Color kunLesungColor = getSuccessOrFailureColor(kunLesungCorrect);

        userInputGermanMeaningSelection.setBackgroundColor(germanMeaningColor);
        userInputOnLesungenSelection.setBackgroundColor(onLesungColor);
        userInputKunLesungenSelection.setBackgroundColor(kunLesungColor);
    }

    private Color getSuccessOrFailureColor(boolean correct) {
        if (correct) {
            return vocableColors.getSuccessColor();
        }
        else {
            return vocableColors.getFailureColor();
        }
    }

    private void fillUserInputTextSelections(String userInputGermanMeaning,
            String userInputOnLesung, String userInputKunLesung) {
        userInputGermanMeaningSelection.setText(userInputGermanMeaning);
        userInputOnLesungenSelection.setText(userInputOnLesung);
        userInputKunLesungenSelection.setText(userInputKunLesung);
    }

    private void initResultTextSelections() {
        SelectionsHelper.initSelectionAsViewer(errorMessageSelection);
        SelectionsHelper.initSelectionAsViewer(memorandumSelection);
    }

    private void fillResultTextSelections() {
        String errorDescription = result.getErrorDescription();
        String memorandum = result.getMemorandum();

        errorMessageSelection.setText(errorDescription);
        memorandumSelection.setText(memorandum);

        errorFilled = !errorDescription.isBlank();
        memorandumFilled = !memorandum.isBlank();
    }

    private void initQuitButton() {
        GuiTools.addReturnListener(quitButton, () -> quitButton.doClick());
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createCenterPart(), BorderLayout.CENTER);
        add(createButtonsPart(),  BorderLayout.SOUTH);

        if (!correct) {
            keybindingsForOnlyTypingError();
        }

        SwingUtilities.invokeLater(() -> quitButton.requestFocus());
        pack();
    }

    private Component createCenterPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(createMessagePart());
        panel.add(createKanjiDataPart());
        panel.add(createUserDataPart());
        if (errorFilled || memorandumFilled) {
            panel.add(createErrorMesageAndMemorandumPart());
        }

        if (!correct) {
            panel.add(createOnlyTypingErrorLabel());
        }

        return GuiTools.createScrollPane(panel);
    }

    private Component createMessagePart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        GuiTools.createTitle(panel);

        panel.add(createMessageLabel(), BorderLayout.CENTER);

        return panel;
    }

    private Component createMessageLabel() {
        JLabel label = new JLabel(message);
        label.setHorizontalAlignment(JLabel.CENTER);
        label.setPreferredSize(new Dimension(MIN_WIDTH, 30));

        GuiTools.biggerFont(label, 7);
        label.setOpaque(true);
        label.setBackground(backgroundColor);
        return label;
    }

    private Component createKanjiDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));
        GuiTools.createBlueTitle("Daten zum abgefragten Kanji:", panel);

        panel.add(kanjiPanel.getPanel());

        return panel;
    }

    private Component createUserDataPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));
        GuiTools.createBlueTitle("Vom Benutzer eingegebene Daten:", panel);

        panel.add(userInputGermanMeaningSelection.getPanel());
        panel.add(userInputOnLesungenSelection.getPanel());
        panel.add(userInputKunLesungenSelection.getPanel());

        return panel;
    }

    private Component createErrorMesageAndMemorandumPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));
        GuiTools.createBlueTitle("Fehler und Hinweise:", panel);

        if (errorFilled) {
            panel.add(errorMessageSelection.getPanel());
        }
        if (memorandumFilled) {
            panel.add(memorandumSelection.getPanel());
        }

        return panel;
    }

    private Component createOnlyTypingErrorLabel() {
        JLabel label = new JLabel("War es nur ein Tippfehler? Dann Strg-Shift-K drücken.");
        label.setBorder(BorderFactory.createEmptyBorder(0,  5,  0,  5));
        return label;
    }

    private Component createButtonsPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.EAST);

        return panel;
    }

    private Component createQuitButton() {
        quitButton.addActionListener(e -> closeDialog());
        quitButton.setBackground(backgroundColor);
        return quitButton;
    }

    private void keybindingsForOnlyTypingError() {
        KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_K,
                InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK);
        String actionMapKey = "Strg-Shift-K";
        Action action = BindKeysOnRootPane.runnableToAction(() -> setToOnlyTypingError());

        JRootPane rootPane = getRootPane();
        InputMap inputMap = rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
        inputMap.put(keyStroke, actionMapKey);
        rootPane.getActionMap().put(actionMapKey, action);
    }

    private void setToOnlyTypingError() {
        String title = "War es nur ein Tippfehler?";
        String userMessage = "War die falsche Eingabe wirklich nur ein Tippfehler?\n"
                + "Bitte gewissenhaft antworten und sich nicht selbst belügen.";
        onlyTypingError = GuiTools.askUser(getWindowAsComponent(), title, userMessage);
        if (onlyTypingError) {
            closeDialog();
        }
    }

    /** Gibt an, ob im Fall einer falsch beantworteten Vokabel nur ein Tippfehler vorlag. */
    public boolean wasOnlyTypingError() {
        return onlyTypingError;
    }

}
