package de.duehl.vocabulary.japanese.ui.dialog.vocables.detail.related;

import java.awt.Point;

import javax.swing.JPanel;

import de.duehl.swing.logic.LongTimeProcessInformer;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.grammar.AdjectivePositivePresenceFinder;
import de.duehl.vocabulary.japanese.grammar.VerbWoerterbuchformFinder;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;

/**
 * Diese Klasse stellt Buttons bereit, um zu einer veränderlichen Vokabel zugehörige Verb- und
 * Adjektivformen zu finden, wenn die Vokabel ein Verb bzw. ein Adjektiv ist.
 *
 * @version 1.01     2025-11-26
 * @author Christian Dühl
 */

public class VerbsAndAdjectivesFinderWithChangingVocableGui {

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Findet zur angezeigten Vokabel gehörige Verb- und Adjektivformen. */
    private final VerbsAndAdjectivesFinder verbsAndAdjectivesFinder;

    /** Der Panel mit den Buttons. */
    private final JPanel buttonPanel;

    /**
     * Konstruktor.
     *
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param location
     *            Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese
     *            Klasse verwendet.
     * @param longTimeProcessInformer
     *            Die Oberfläche, welche diese Klasse aufruft, auf der man eine GlassPane anzeigen
     *            kann.
     */
    public VerbsAndAdjectivesFinderWithChangingVocableGui(FumikoDataStructures dataStructures,
            FumikoUiObjects uiObjects, Point location,
            LongTimeProcessInformer longTimeProcessInformer) {
        this.dataStructures = dataStructures;

        verbsAndAdjectivesFinder = new VerbsAndAdjectivesFinder(dataStructures, uiObjects, location,
                longTimeProcessInformer);

        buttonPanel = new JPanel();
        buttonPanel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));
    }

    /**
     * Fügt zum übergebenen Panel Buttons zur Anzeige der Verb- bzw. Adjektivformen an, abhängig
     * davon, ob die Vokabel ein Verb bzw. Adjektiv ist.
     *
     * @param panel
     *            Der Panel dem die Buttons hinzugefügt werden sollen. Dieser sollte ein Layout
     *            haben, das das Hinzufügen von Elemente ohne weitere Parameter gestattet, wie z.B.
     *            das VerticalLayout.
     */
    public void appendButtonsToPanel(JPanel panel) {
        panel.add(buttonPanel);
    }

    /** Setter für die Vokabel. */
    public void setVocable(Vocable vocable) {
        updateButtonPanel(vocable);
    }

    private void updateButtonPanel(Vocable vocable) {
        VerbAndAdjectiveButtonsCreator creator = new VerbAndAdjectiveButtonsCreator(vocable,
                verbsAndAdjectivesFinder);

        buttonPanel.removeAll();

        if (vocable.isVerbInWoerterbuchform()) {
            buttonPanel.add(creator.createFindRelatedVerbsButton());
        }
        else if (vocable.isVerb()) {
            Vocable woerterbuchform = VerbWoerterbuchformFinder.determineWoerterbuchform(vocable,
                    dataStructures.getVocabularies());
            if (!woerterbuchform.equals(VerbWoerterbuchformFinder.NO_WOERTERBUCHFORM_FOUND)) {
                buttonPanel.add(
                        creator.createFindRelatedVerbsButtonViaWoerterbuchform(woerterbuchform));
            }
        }
        else if (vocable.isAdjectivInPositivePresence()) {
            buttonPanel.add(creator.createFindRelatedAdjectiveButton());
        }
        else if (vocable.isAdjective()) {
            Vocable positivePresence = AdjectivePositivePresenceFinder.determinePositivePresence(
                    vocable, dataStructures.getVocabularies());
            if (!positivePresence.equals(
                    AdjectivePositivePresenceFinder.NO_POSITIVE_PRESENCE_FOUND)) {
                buttonPanel.add(
                        creator.createFindRelatedAdjectiveButtonViaPositivePresence(
                                positivePresence));
            }
        }
    }

}
