package de.duehl.vocabulary.japanese.website.update.program;

import java.awt.Component;
import java.awt.Image;
import java.awt.Point;

import de.duehl.basics.version.Version;
import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.website.update.program.ProgramUpdaterTools;

import static de.duehl.vocabulary.japanese.common.website.download.WebsiteFileContentDownloader.*;
import static de.duehl.vocabulary.japanese.common.website.update.UpdateTools.informUserAboutDownloadFailure;

/**
 * Diese Klasse prüft aus dem Vokabeltrainer selbst heraus, ob neuere Versionen des Programms (des
 * Vokabeltrainers) vorliegen.
 *
 * Diese Klasse wird aus dem Programm aufgerufen, nicht aus dem Launcher und auch nicht zum Startup
 * des Trainers, da das bereits der Launcher erledigt.
 *
 * Vergleiche Klasse
 *     de.duehl.vocabulary.japanese.launcher.logic.LauncherLogic
 *
 * @version 1.01     2024-12-11
 * @author Christian Dühl
 */

public class ProgramUpdatableChecker {

    /**
     * Die Version des laufenden Trainers, falls von diesem aufgerufen wird, oder des neusten
     * vorliegenden Trainers, falls vom Launcher aufgerufen wird.
     */
    private final String versionNumberRunning;

    /** Die aufrufende Gui als Komponente zur Positionierung von Dialogen. */
    private final Component guiComponent;

    /** Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird. */
    private final Point parentLocation;

    /** Das anzuzeigende ProgrammIcon. */
    private final Image programImage;

    /**
     * Konstruktor.
     *
     * @param versionNumberRunning
     *            Die Version des laufenden Trainers, falls von diesem aufgerufen wird, oder des
     *            neusten vorliegenden Trainers, falls vom Launcher aufgerufen wird.
     * @param guiComponent
     *            Die aufrufende Gui als Komponente zur Positionierung von Dialogen.
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog erzeugt wird.
     * @param programImage
     *            Das anzuzeigende ProgrammIcon.
     */
    public ProgramUpdatableChecker(String versionNumberRunning,
            Component guiComponent, Point parentLocation, Image programImage) {
        this.versionNumberRunning = versionNumberRunning;
        this.guiComponent = guiComponent;
        this.parentLocation = parentLocation;
        this.programImage = programImage;
    }

    /** Prüft auf eine aktuellere Version des Programms. */
    public void update() {
        String versionNumberOnServer = ProgramUpdaterTools.downloadProgramVersionFromServer();
        if (versionNumberOnServer.equals(NO_DOWNLOAD_SUCCESS)) {
            informUserAboutDownloadFailure("des Programms", parentLocation, programImage);
        }
        else {
            Version versionOnServer = new Version(versionNumberOnServer, "interessiert nicht");
            Version versionRunning = new Version(versionNumberRunning, "interessiert nicht");
            if (versionOnServer.isNewerThan(versionRunning)) {
                informUserAboutNewerProgramVersion(versionNumberRunning, versionNumberOnServer);
            }
            else {
                informUserAboutProgramVersionIsActual(versionNumberRunning);
            }
        }
    }

    private void informUserAboutNewerProgramVersion(String verionNumberRunning,
            String versionNumberOnServer) {
        String title = "Es gibt eine neuere Programmversion";
        String message = "Es gibt eine neuere Programmversion!\n\n"
                + "Aktuelle Programmversion: " + verionNumberRunning + "\n"
                + "Neuere Programmversion: " + versionNumberOnServer + "\n\n"
                + "Bitte beenden sie den Trainer und starten Sie den Launcher, um die neue Version "
                + "herunterzuladen.\n"
                + "Dafür muss das Programm nur neu getsartet werden, dabei startet zuerst der "
                + "Launcher und anschließend der Vokabeltrainer.";
        GuiTools.informUserInEdt(title, message);
    }

    private void informUserAboutProgramVersionIsActual(String verionNumberRunning) {
        String title = "Der Vokabeltrainer ist aktuell";
        String message = "Die verwendete Programmversion " + verionNumberRunning + " ist aktuell.";
        GuiTools.informUserInEdt(guiComponent, title, message);
    }

}
