package de.duehl.vocabulary.japanese.data.symbol;

import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.*;

import java.util.List;

public class KanjiTest {

    @Test
    public void containsKanjiEmptyInput() {
        String text = "";
        assertFalse(Kanji.containsKanji(text));
    }

    @Test
    public void containsKanjiNoKanjiInput() {
        String text = "abc あ ア";
        assertFalse(Kanji.containsKanji(text));
    }

    @Test
    public void containsKanjiKanjiInput() {
        String text = "大好き (だいすき) wirklich lieben";
        assertTrue(Kanji.containsKanji(text));
    }

    @Test
    public void containsKanjiOnlyKanjiInput() {
        String text = "大好";
        assertTrue(Kanji.containsKanji(text));
    }

    @Test
    public void containscontainsOnlyKanjiEmptyInput() {
        String text = "";
        assertFalse(Kanji.containsOnlyKanji(text));
    }

    @Test
    public void containsOnlyKanjiNoKanjiInput() {
        String text = "abc あ ア";
        assertFalse(Kanji.containsOnlyKanji(text));
    }

    @Test
    public void containsOnlyKanjiKanjiInput() {
        String text = "大好き (だいすき) wirklich lieben";
        assertFalse(Kanji.containsOnlyKanji(text));
    }

    @Test
    public void containsOnlyKanjiOnlyKanjiInput() {
        String text = "大好";
        assertTrue(Kanji.containsOnlyKanji(text));
    }

    @Test
    public void firstIndexOfKanjiEmptyText() {
        String text = "";
        assertEquals(-1, Kanji.firstIndexOfKanji(text));
    }

    @Test
    public void firstIndexOfKanjiNoKanji() {
        String text = "    (だいすき) wirklich lieben";
        assertEquals(-1, Kanji.firstIndexOfKanji(text));
    }

    @Test
    public void firstIndexOfKanjiStartingWithKanji() {
        String text = "大好き (だいすき) wirklich lieben";
        assertEquals(0, Kanji.firstIndexOfKanji(text));
    }

    @Test
    public void firstIndexOfKanjiContainingWithKanji() {
        String text = "Text: 大好き (だいすき) wirklich lieben";
        assertEquals(6, Kanji.firstIndexOfKanji(text));
    }

    @Test
    public void firstIndexOfKanjiOnlyKanji() {
        String text = "大好";
        assertEquals(0, Kanji.firstIndexOfKanji(text));
    }

    @Test
    public void lastIndexOfKanjiEmptyText() {
        String text = "";
        assertEquals(-1, Kanji.lastIndexOfKanji(text));
    }

    @Test
    public void lastIndexOfKanjiNoKanji() {
        String text = "    (だいすき) wirklich lieben";
        assertEquals(-1, Kanji.lastIndexOfKanji(text));
    }

    @Test
    public void lastIndexOfKanjiStartingWithKanji() {
        String text = "大好き (だいすき) wirklich lieben";
        assertEquals(1, Kanji.lastIndexOfKanji(text));
    }

    @Test
    public void lastIndexOfKanjiContainingWithKanji() {
        String text = "Text: 大好き (だいすき) wirklich lieben";
        assertEquals(7, Kanji.lastIndexOfKanji(text));
    }

    @Test
    public void lastIndexOfKanjiOnlyKanji() {
        String text = "大好";
        assertEquals(1, Kanji.lastIndexOfKanji(text));
    }

    @Test
    public void getByCharacter() {
        String character = "好";
        Kanji actual = Kanji.getByCharacter(character);
        Kanji expected = Kanji.MOEGEN_LIEBEN;
        assertEquals(expected, actual);
    }


    @Test
    public void getAllKanjiFromTextEmptyText() {
        String text = "";
        List<Kanji> kanjiInText = Kanji.getAllKanjiFromText(text);
        assertEquals(0, kanjiInText.size());
    }

    @Test
    public void getAllKanjiFromTextNoKanji() {
        String text = "    (だいすき) wirklich lieben";
        List<Kanji> kanjiInText = Kanji.getAllKanjiFromText(text);
        assertEquals(0, kanjiInText.size());
    }

    @Test
    public void getAllKanjiFromTextStartingWithKanji() {
        String text = "大好き (だいすき) wirklich lieben";
        List<Kanji> kanjiInText = Kanji.getAllKanjiFromText(text);
        assertEquals(2, kanjiInText.size());
        assertEquals(Kanji.GROSS, kanjiInText.get(0));
        assertEquals(Kanji.MOEGEN_LIEBEN, kanjiInText.get(1));
    }

    @Test
    public void getAllKanjiFromTextContainingWithKanji() {
        String text = "Text: 大好き (だいすき) wirklich lieben";
        List<Kanji> kanjiInText = Kanji.getAllKanjiFromText(text);
        assertEquals(2, kanjiInText.size());
        assertEquals(Kanji.GROSS, kanjiInText.get(0));
        assertEquals(Kanji.MOEGEN_LIEBEN, kanjiInText.get(1));
    }

    @Test
    public void getAllKanjiFromTextOnlyKanji() {
        String text = "大好";
        List<Kanji> kanjiInText = Kanji.getAllKanjiFromText(text);
        assertEquals(2, kanjiInText.size());
        assertEquals(Kanji.GROSS, kanjiInText.get(0));
        assertEquals(Kanji.MOEGEN_LIEBEN, kanjiInText.get(1));
    }

    @Test
    public void getAllKanjiFromTextContainingKanjiMultipleTimes() {
        String text = "Text: 大好き (だいすき) wirklich lieben und 大嫌い (だいきらい) wirklich hassen";
        List<Kanji> kanjiInText = Kanji.getAllKanjiFromText(text);
        assertEquals(3, kanjiInText.size()); // TODO später 4,  嫌 noch nicht enthalten!
        assertEquals(Kanji.GROSS, kanjiInText.get(0));
        assertEquals(Kanji.MOEGEN_LIEBEN, kanjiInText.get(1));
        assertEquals(Kanji.GROSS, kanjiInText.get(2));
        //assertEquals(Kanji.ABLEHNEN, kanjiInText.get(3)); // TODO 嫌 noch nicht enthalten!
    }

}
