package de.duehl.vocabulary.japanese.website.update.ownlists;

import org.junit.jupiter.api.Test;

import de.duehl.vocabulary.japanese.website.update.ownlists.data.OwnListInterestAndVersionEntry;
import de.duehl.vocabulary.japanese.website.update.ownlists.parser.OwnListVersionAndInterestParser;

import static org.junit.jupiter.api.Assertions.*;

import java.util.List;

public class OwnListVersionAndInterestParserTest {

    @Test
    void emptyFileTest() {
        String content = """
                """;

        OwnListVersionAndInterestParser parser = new OwnListVersionAndInterestParser(content);
        parser.parse();
        assertTrue(parser.isSuccess());
        assertEquals("", parser.getFailReason());
        List<OwnListInterestAndVersionEntry> entries = parser.getOwnListInterestAndVersionEntries();
        assertEquals(0, entries.size());
    }

    @Test
    void normalCase() {
        String content = """
                FROM_ZERO_1 - nein - 0
                FROM_ZERO_2 - nein - 1
                FROM_ZERO_3 - nein - 2
                SCHRITT_FUER_SCHRITT_1 - nein - 3
                VHS_A1_1 - ja -  4
                VHS_A1_2 - ja -  5
                MARUGOTO_A1 - ja -  6
                """;

        OwnListVersionAndInterestParser parser = new OwnListVersionAndInterestParser(content);
        parser.parse();
        assertTrue(parser.isSuccess());
        assertEquals("", parser.getFailReason());
        List<OwnListInterestAndVersionEntry> entries = parser.getOwnListInterestAndVersionEntries();
        assertEquals(7, entries.size());

        OwnListInterestAndVersionEntry entry1 = entries.get(0);
        OwnListInterestAndVersionEntry entry2 = entries.get(1);
        OwnListInterestAndVersionEntry entry3 = entries.get(2);
        OwnListInterestAndVersionEntry entry4 = entries.get(3);
        OwnListInterestAndVersionEntry entry5 = entries.get(4);
        OwnListInterestAndVersionEntry entry6 = entries.get(5);
        OwnListInterestAndVersionEntry entry7 = entries.get(6);

        assertEquals("FROM_ZERO_1", entry1.getGroupName());
        assertFalse(entry1.isInterested());
        assertEquals(0, entry1.getVersion());

        assertEquals("FROM_ZERO_2", entry2.getGroupName());
        assertFalse(entry2.isInterested());
        assertEquals(1, entry2.getVersion());

        assertEquals("FROM_ZERO_3", entry3.getGroupName());
        assertFalse(entry3.isInterested());
        assertEquals(2, entry3.getVersion());

        assertEquals("SCHRITT_FUER_SCHRITT_1", entry4.getGroupName());
        assertFalse(entry4.isInterested());
        assertEquals(3, entry4.getVersion());

        assertEquals("VHS_A1_1", entry5.getGroupName());
        assertTrue(entry5.isInterested());
        assertEquals(4, entry5.getVersion());

        assertEquals("VHS_A1_2", entry6.getGroupName());
        assertTrue(entry6.isInterested());
        assertEquals(5, entry6.getVersion());

        assertEquals("MARUGOTO_A1", entry7.getGroupName());
        assertTrue(entry7.isInterested());
        assertEquals(6, entry7.getVersion());
    }

    @Test
    void notThreeParts1() {
        String content = """
                FROM_ZERO_1 - nein - 0
                FROM_ZERO_2 - nein - 0
                FROM_ZERO_3 - nein - 0
                SCHRITT_FUER_SCHRITT_1 - nein - 0 - bla
                VHS_A1_1 - ja -  0
                VHS_A1_2 - ja -  0
                MARUGOTO_A1 - ja -  0
                """;

        OwnListVersionAndInterestParser parser = new OwnListVersionAndInterestParser(content);
        parser.parse();
        assertFalse(parser.isSuccess());
        assertEquals("Die Zeile 'SCHRITT_FUER_SCHRITT_1 - nein - 0 - bla' lässt sich nicht parsen: "
                + "Die Zeile lässt sich nicht an '-' in drei Teile teilen.",
                parser.getFailReason());
        List<OwnListInterestAndVersionEntry> entries = parser.getOwnListInterestAndVersionEntries();
        assertEquals(0, entries.size());
    }

    @Test
    void notThreeParts2() {
        String content = """
                FROM_ZERO_1 - nein - 0
                FROM_ZERO_2 - nein - 0
                FROM_ZERO_3 - nein - 0
                SCHRITT_FUER_SCHRITT_1 - nein . 0
                VHS_A1_1 - ja -  0
                VHS_A1_2 - ja -  0
                MARUGOTO_A1 - ja -  0
                """;

        OwnListVersionAndInterestParser parser = new OwnListVersionAndInterestParser(content);
        parser.parse();
        assertFalse(parser.isSuccess());
        assertEquals("Die Zeile 'SCHRITT_FUER_SCHRITT_1 - nein . 0' lässt sich nicht parsen: "
                + "Die Zeile lässt sich nicht an '-' in drei Teile teilen.",
                parser.getFailReason());
        List<OwnListInterestAndVersionEntry> entries = parser.getOwnListInterestAndVersionEntries();
        assertEquals(0, entries.size());
    }

    @Test
    void versionNotOnlyDigits() {
        String content = """
                FROM_ZERO_1 - nein - 0
                FROM_ZERO_2 - nein - 0
                FROM_ZERO_3 - nein - 0
                SCHRITT_FUER_SCHRITT_1 - nein - 0
                VHS_A1_1 - ja -  0
                VHS_A1_2 - ja -  0
                MARUGOTO_A1 - ja -  1.1
                """;

        OwnListVersionAndInterestParser parser = new OwnListVersionAndInterestParser(content);
        parser.parse();
        assertFalse(parser.isSuccess());
        assertEquals("Die Zeile 'MARUGOTO_A1 - ja -  1.1' lässt sich nicht parsen: "
                + "Die Version '1.1' in der Zeile besteht nicht nur aus Ziffern",
                parser.getFailReason());
        List<OwnListInterestAndVersionEntry> entries = parser.getOwnListInterestAndVersionEntries();
        assertEquals(0, entries.size());
    }

    @Test
    void interestFail() {
        String content = """
                FROM_ZERO_1 - nein - 0
                FROM_ZERO_2 - nein - 0
                FROM_ZERO_3 - nein - 0
                SCHRITT_FUER_SCHRITT_1 - nein - 0
                VHS_A1_1 - weiß nicht -  0
                VHS_A1_2 - ja -  0
                MARUGOTO_A1 - ja -  1
                """;

        OwnListVersionAndInterestParser parser = new OwnListVersionAndInterestParser(content);
        parser.parse();
        assertFalse(parser.isSuccess());
        assertEquals("Die Zeile 'VHS_A1_1 - weiß nicht -  0' lässt sich nicht parsen: "
                + "Der Wert des Strings der einen Booleanwert darstellen soll 'weiß nicht' ist "
                + "unbekannt.",
                parser.getFailReason());
        List<OwnListInterestAndVersionEntry> entries = parser.getOwnListInterestAndVersionEntries();
        assertEquals(0, entries.size());
    }

}
